<?php

namespace App\Traits;

use App\Models\Deposit;
use App\Models\QrCode;
use App\Models\Transaction;
use App\Models\Withdrawal;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

trait UserPartials
{

    public function createQrCode()
    {

        $guard    = userGuard();
        $user     = $guard['user'];
        $userType = $guard['type'];
        $qrCode   = $user->qrCode;

        if (!$qrCode) {
            $qrCode              = new QrCode();
            $qrCode->user_id     = $user->id;
            $qrCode->user_type   = $userType;
            $qrCode->unique_code = keyGenerator(15);
            $qrCode->save();
        }

        return $qrCode;
    }

    public function downLoadQrCode()
    {
        $guard = userGuard();
        $user  = $guard['user'];

        $qrCode  = $user->qrCode()->first();
        $general = gs();

        $file     = cryptoQR($qrCode->unique_code);
        $filename = $qrCode->unique_code . '.jpg';

        $template = 'default.png';
        if ($general->qr_code_template) {
            $template = $general->qr_code_template;
        }

        $manager  = new ImageManager(new Driver());
        $template = $manager->read('assets/images/qr_code_template/' . $template);

        $client       = new Client();
        $response     = $client->get($file);
        $imageContent = $response->getBody()->getContents();

        $qrCode = $manager->read($imageContent)->cover(2000, 2000);
        $template->place($qrCode, 'center');
        $image   = $template->encode();
        $headers = [
            'Content-Type'        => 'image/jpeg',
            'Content-Disposition' => 'attachment; filename=' . $filename,
        ];

        return response()->stream(function () use ($image) {
            echo $image;
        }, 200, $headers);
    }

    public function cashInOut($day = 7)
    {
        $guard    = userGuard();
        $userType = $guard['type'];
        $scope    = 'for' . ucfirst(strtolower($userType));

        $date    = Carbon::today()->subDays($day);
        $cashIn  = Transaction::$scope()->where('trx_type', '+')->whereDate('created_at', '>=', $date)->sum('amount');
        $cashOut = Transaction::$scope()->where('trx_type', '-')->whereDate('created_at', '>=', $date)->sum('amount');

        return ['totalCashIn' => $cashIn, 'totalCashOut' => $cashOut];
    }

    public function trxLimit($type, $user = null)
    {
        if ($user == null) {
            $guard        = userGuard();
            $user         = $guard['user'];
        }

        $transactions = $user->transactions()->where('remark', $type)->selectRaw("SUM(amount) as totalAmount");

        return [
            'daily'   => $transactions->whereDate('transactions.created_at', Carbon::now())->get()->sum('totalAmount'),
            'monthly' => $transactions->whereMonth('transactions.created_at', Carbon::now())->whereYear('transactions.created_at', Carbon::now())->get()->sum('totalAmount'),
        ];
    }

    public function trxLog($request)
    {
        $guard    = userGuard();
        $user     = $guard['user'];
        $userType = $guard['type'];

        $search    = $request->search;
        $type      = $request->type;
        $operation = $request->operation;

        if ($type && $type == 'plus_trx') {
            $type = '+';
        } elseif ($type && $type == 'minus_trx') {
            $type = '-';
        }

        $time = $request->time;
        if ($time) {
            if ($time == '7days') {
                $time = 7;
            } elseif ($time == '15days') {
                $time = 15;
            } elseif ($time == '30days') {
                $time = 30;
            } elseif ($time == '365days') {
                $time = 365;
            }
        }

        $histories = Transaction::where('user_id', $user->id)->where('user_type', $userType)
            ->when($search, function ($trx, $search) {
                return $trx->where('trx', $search);
            })
            ->when($type, function ($trx, $type) {
                return $trx->where('trx_type', $type);
            })
            ->when($time, function ($trx, $time) {
                return $trx->where('created_at', '>=', Carbon::today()->subDays($time));
            })
            ->when($operation, function ($trx, $operation) {
                return $trx->where('remark', $operation);
            })
            ->with(['receiverUser', 'receiverAgent', 'receiverMerchant'])->orderBy('id', 'DESC')->paginate(getPaginate());

        return $histories;
    }

    public function totalDeposit()
    {

        $guard    = userGuard();
        $userType = $guard['type'];
        $scope    = 'for' . ucfirst(strtolower($userType));

        $log = Deposit::$scope()->successful()->selectRaw('SUM(amount) as totalAmount')->first();
        return $log->totalAmount;
    }

    public function totalWithdraw()
    {
        $guard    = userGuard();
        $userType = $guard['type'];
        $scope    = 'for' . ucfirst(strtolower($userType));

        $log = Withdrawal::$scope()->approved()->selectRaw('SUM(amount) as totalAmount')->first();
        return $log->totalAmount;
    }

    public function trxGraph()
    {

        $guard    = userGuard();
        $userType = $guard['type'];
        $scope    = 'for' . ucfirst(strtolower($userType));

        // Transaction Graph
        $report['trx_dates']  = collect([]);
        $report['trx_amount'] = collect([]);

        $transactions = Transaction::$scope()
            ->where('trx_type', '+')
            ->where('transactions.created_at', '>=', Carbon::now()->subYear())
            ->selectRaw("SUM(amount) as totalAmount")
            ->selectRaw("DATE_FORMAT(transactions.created_at,'%M-%Y') as dates")
            ->orderBy('transactions.created_at')
            ->groupBy('dates')
            ->get();

        $transactions->map(function ($trxData) use ($report) {
            $report['trx_dates']->push($trxData->dates);
            $report['trx_amount']->push($trxData->totalAmount);
        });

        return $report;
    }
}
