<?php

namespace App\Providers;

use App\Models\User;
use App\Models\Agent;
use App\Lib\Searchable;
use App\Models\Deposit;
use App\Models\Frontend;
use App\Models\Merchant;
use App\Constants\Status;
use App\Models\Withdrawal;
use App\Models\UtilityBill;
use App\Models\BankTransfer;
use App\Models\ModuleSetting;
use App\Models\SupportTicket;
use App\Models\MobileRecharge;
use App\Models\AdminNotification;
use App\Models\SavingAccount;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\Eloquent\Builder;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        Builder::mixin(new Searchable);
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        if (!cache()->get('SystemInstalled')) {
            $envFilePath = base_path('.env');
            if (!file_exists($envFilePath)) {
                header('Location: install');
                exit;
            }
            $envContents = file_get_contents($envFilePath);
            if (empty($envContents)) {
                header('Location: install');
                exit;
            } else {
                cache()->put('SystemInstalled', true);
            }
        }

        $viewShare['emptyMessage']       = 'Data not found';
        $viewShare['module']             = ModuleSetting::get();
        view()->share($viewShare);

        view()->composer('admin.partials.sidenav', function ($view) {
            $view->with([
                'bannedUsersCount'               => User::banned()->count(),
                'emailUnverifiedUsersCount'      => User::emailUnverified()->count(),
                'mobileUnverifiedUsersCount'     => User::mobileUnverified()->count(),
                'kycUnverifiedUsersCount'        => User::kycUnverified()->count(),
                'kycPendingUsersCount'           => User::kycPending()->count(),

                'bannedAgentsCount'              => Agent::banned()->count(),
                'emailUnverifiedAgentsCount'     => Agent::emailUnverified()->count(),
                'mobileUnverifiedAgentsCount'    => Agent::mobileUnverified()->count(),
                'kycUnverifiedAgentsCount'       => Agent::kycUnverified()->count(),
                'kycPendingAgentsCount'          => Agent::kycPending()->count(),

                'bannedMerchantsCount'           => Merchant::banned()->count(),
                'emailUnverifiedMerchantsCount'  => Merchant::emailUnverified()->count(),
                'mobileUnverifiedMerchantsCount' => Merchant::mobileUnverified()->count(),
                'kycUnverifiedMerchantsCount'    => Merchant::kycUnverified()->count(),
                'kycPendingMerchantsCount'       => Merchant::kycPending()->count(),

                'pendingTicketCount'             => SupportTicket::whereIN('status', [Status::TICKET_OPEN, Status::TICKET_REPLY])->count(),
                'pendingDepositsCount'           => Deposit::pending()->count(),
                'pendingWithdrawCount'           => Withdrawal::pending()->count(),

                'pendingMobileRechargeCount'     => MobileRecharge::pending()->count(),
                'pendingUtilityBillCount'        => UtilityBill::pending()->count(),
                'pendingBankTransferCount'       => BankTransfer::pending()->count(),

                'updateAvailable'                => version_compare(gs('available_version'), systemDetails()['version'], '>') ? 'v' . gs('available_version') : false,
            ]);
        });

        view()->composer('admin.partials.topnav', function ($view) {
            $view->with([
                'adminNotifications'     => AdminNotification::where('is_read', Status::NO)->with('user')->orderBy('id', 'desc')->take(10)->get(),
                'adminNotificationCount' => AdminNotification::where('is_read', Status::NO)->count(),
            ]);
        });

        view()->composer('partials.seo', function ($view) {
            $seo = Frontend::where('data_keys', 'seo.data')->first();
            $view->with([
                'seo' => $seo ? $seo->data_values : $seo,
            ]);
        });

        if (gs('force_ssl')) {
            \URL::forceScheme('https');
        }

        Paginator::useBootstrapFive();
    }
}
