<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\UserNotify;
use App\Traits\UserPartials;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, UserNotify, UserPartials;

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'ver_code',
        'kyc_data',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'kyc_data'          => 'object',
        'ver_code_send_at'  => 'datetime',
    ];

    public function __construct()
    {
        if (request()->is('api/*')) {
            $this->appends[] = 'get_image';
        }
    }

    public function getGetImageAttribute()
    {
        if ($this->image) {
            return getImage(getFilePath('userProfile') . '/' . $this->image, getFileSize('userProfile'));
        } else {
            return asset('assets/images/profile.png');
        }
    }

    public function getIsPinSetAttribute()
    {
        if ($this->pin) {
            return Status::YES;
        }

        return Status::NO;
    }

    public function qrCode()
    {
        return $this->hasOne(QrCode::class, 'user_id')->where('user_type', 'USER');
    }

    public function loginLogs()
    {
        return $this->hasMany(UserLogin::class);
    }

    public function points()
    {
        return $this->hasMany(PointLog::class, 'user_id')->latest();
    }

    public function nominees()
    {
        return $this->hasMany(Nominee::class);
    }

    public function savingAccounts()
    {
        return $this->hasMany(SavingAccount::class);
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class)->where('user_type', 'USER')->orderBy('id', 'desc');
    }

    public function deposits()
    {
        return $this->hasMany(Deposit::class)->where('user_type', 'USER')->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function tickets()
    {
        return $this->hasMany(SupportTicket::class)->where('user_type', 'USER');
    }

    public function fullname(): Attribute
    {
        return new Attribute(
            get: fn() => $this->firstname . ' ' . $this->lastname,
        );
    }

    public function mobileNumber(): Attribute
    {
        return new Attribute(
            get: fn() => $this->dial_code . $this->mobile,
        );
    }

    public function getPassedLevelsAttribute()
    {
        $totalPoints = $this->points->sum('points');
        return getLevel($totalPoints);
    }

    public function redeemRewards()
    {
        return $this->hasMany(RedeemReward::class)->orderBy('id', 'asc');
    }

    // SCOPES
    public function scopeActive($query)
    {
        return $query->where('status', Status::USER_ACTIVE)->where('ev', Status::VERIFIED)->where('sv', Status::VERIFIED);
    }

    public function scopeBanned($query)
    {
        return $query->where('status', Status::USER_BAN);
    }

    public function scopeEmailUnverified($query)
    {
        return $query->where('ev', Status::UNVERIFIED);
    }

    public function scopeMobileUnverified($query)
    {
        return $query->where('sv', Status::UNVERIFIED);
    }

    public function scopeKycUnverified($query)
    {
        return $query->where('kv', Status::KYC_UNVERIFIED);
    }

    public function scopeKycPending($query)
    {
        return $query->where('kv', Status::KYC_PENDING);
    }

    public function scopeEmailVerified($query)
    {
        return $query->where('ev', Status::VERIFIED);
    }

    public function scopeMobileVerified($query)
    {
        return $query->where('sv', Status::VERIFIED);
    }

    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }

    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class)->where('user_type', 'USER');
    }

    public function mobileRecharges()
    {
        return $this->hasMany(MobileRecharge::class);
    }

    public function bankTransfers()
    {
        return $this->hasMany(BankTransfer::class);
    }

    public function utilityBills()
    {
        return $this->hasMany(UtilityBill::class);
    }
}
