<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\UserPartials;
use App\Traits\MerchantNotify;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Merchant extends Authenticatable
{

    use HasApiTokens, MerchantNotify, HasFactory, UserPartials;

    protected $table = "merchants";

    protected $hidden = ['password'];

    protected $guarded = [];

    protected $appends = [];

    public function __construct()
    {
        if (request()->is('api/*')) {
            $this->appends[] = 'get_image';
        }
    }

    public function getGetImageAttribute()
    {
        if ($this->image) {
            return getImage(getFilePath('merchantProfile') . '/' . $this->image, getFileSize('merchantProfile'));
        } else {
            return asset('assets/images/profile.png');
        }
    }

    protected $casts = [
        'email_verified_at' => 'datetime',
        'address'           => 'object',
        'ver_code_send_at'  => 'datetime',
        'kyc_data'          => 'object',
    ];

    public function getFullnameAttribute()
    {
        return $this->firstname . ' ' . $this->lastname;
    }

    public function mobileNumber(): Attribute
    {
        return new Attribute(
            get: fn () => $this->mobile,
        );
    }

    public function qrCode()
    {
        return $this->hasOne(QrCode::class, 'user_id')->where('user_type', 'MERCHANT');
    }

    public function loginLogs()
    {
        return $this->hasMany(UserLogin::class, 'merchant_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'user_id')->where('user_type', 'MERCHANT')->orderBy('id', 'desc');
    }

    public function withdrawals()
    {
        return $this->hasMany(Withdrawal::class, 'user_id')->where('user_type', 'MERCHANT')->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function tickets()
    {
        return $this->hasMany(SupportTicket::class, 'user_id')->where('user_type', 'MERCHANT');
    }

    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class, 'user_id')->where('user_type', 'MERCHANT');
    }

    public function scopeActive($query)
    {
        return $query->where('status', Status::USER_ACTIVE)->where('ev', Status::VERIFIED)->where('sv', Status::VERIFIED);
    }

    public function scopeBanned($query)
    {
        return $query->where('status', Status::USER_BAN);
    }

    public function scopeEmailUnverified($query)
    {
        return $query->where('ev', Status::UNVERIFIED);
    }

    public function scopeKycUnverified($query)
    {
        return $query->where('kv', Status::KYC_UNVERIFIED);
    }

    public function scopeKycPending($query)
    {
        return $query->where('kv', Status::KYC_PENDING);
    }

    public function scopeMobileUnverified($query)
    {
        return $query->where('sv', Status::UNVERIFIED);
    }

    public function scopeEmailVerified($query)
    {
        return $query->where('ev', Status::VERIFIED);
    }

    public function scopeSmsVerified($query)
    {
        return $query->where('sv', Status::VERIFIED);
    }

    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }
}
