<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\AgentNotify;
use App\Traits\UserPartials;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Agent extends Authenticatable
{

    use HasFactory, AgentNotify, UserPartials, HasApiTokens;

    protected $table = "agents";

    protected $hidden = ['password'];

    protected $appends = [];

    public function __construct()
    {
        if (request()->is('api/*')) {
            $this->appends[] = 'get_image';
        }
    }

    public function getGetImageAttribute()
    {
        if ($this->image) {
            return getImage(getFilePath('agentProfile') . '/' . $this->image, getFileSize('agentProfile'));
        } else {
            return asset('assets/images/profile.png');
        }
    }

    protected $casts = [
        'email_verified_at' => 'datetime',
        'address'           => 'object',
        'kyc_data'          => 'object',
        'ver_code_send_at'  => 'datetime',
    ];

    protected $guarded = [];

    public function getFullnameAttribute()
    {
        return $this->firstname . ' ' . $this->lastname;
    }

    public function mobileNumber(): Attribute
    {
        return new Attribute(
            get: fn () => $this->mobile,
        );
    }

    public function qrCode()
    {
        return $this->hasOne(QrCode::class, 'user_id')->where('user_type', 'AGENT');
    }

    public function loginLogs()
    {
        return $this->hasMany(UserLogin::class, 'agent_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'user_id')->where('user_type', 'AGENT')->orderBy('id', 'desc');
    }

    public function deposits()
    {
        return $this->hasMany(Deposit::class, 'user_id')->where('user_type', 'AGENT')->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function withdrawals()
    {
        return $this->hasMany(Withdrawal::class, 'user_id')->where('user_type', 'AGENT')->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function tickets()
    {
        return $this->hasMany(SupportTicket::class, 'user_id')->where('user_type', 'AGENT');
    }

    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class, 'user_id')->where('user_type', 'AGENT');
    }

    public function scopeActive($query)
    {
        return $query->where('status', Status::USER_ACTIVE)->where('ev', Status::VERIFIED)->where('sv', Status::VERIFIED);
    }

    public function scopeBanned($query)
    {
        return $query->where('status', Status::USER_BAN);
    }

    public function scopeEmailUnverified($query)
    {
        return $query->where('ev', Status::UNVERIFIED);
    }

    public function scopeMobileUnverified($query)
    {
        return $query->where('sv', Status::UNVERIFIED);
    }

    public function scopeEmailVerified($query)
    {
        return $query->where('ev', Status::VERIFIED);
    }

    public function scopeSmsVerified($query)
    {
        return $query->where('sv', Status::VERIFIED);
    }

    public function scopeKycUnverified($query)
    {
        return $query->where('kv', Status::KYC_UNVERIFIED);
    }

    public function scopeKycPending($query)
    {
        return $query->where('kv', Status::KYC_PENDING);
    }

    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }
}
