<?php

namespace App\Http\Controllers\Merchant;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\TransactionCharge;
use App\Models\User;
use Illuminate\Http\Request;

class InvoiceController extends Controller {

    public function index() {
        $pageTitle = "Invoice All";
        $invoices  = Invoice::searchable(['uid', 'user:username', 'user:mobile', 'user:email'])->with('invoiceItem')->where('merchant_id', merchant()->id)->latest()->paginate(getPaginate());
        return view('Template::merchant.invoice.index', compact('pageTitle', 'invoices'));
    }

    public function create() {
        $pageTitle     = "New Invoice";
        $paymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        return view('Template::merchant.invoice.create', compact('pageTitle', 'paymentCharge'));
    }

    public function edit($id) {
        $invoice       = Invoice::unpaid()->withCount('invoiceItem')->with(['invoiceItem', 'user'])->where('merchant_id', merchant()->id)->findOrFail($id);
        $pageTitle     = "Edit Invoice - " . '#' . $invoice->uid;
        $paymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        return view('Template::merchant.invoice.create', compact('pageTitle', 'invoice', 'paymentCharge'));
    }

    public function store(Request $request, $id = null) {
        $userValidation = $id ? 'nullable' : 'required';

        $request->validate([
            'name'     => 'required|array',
            'name.*'   => 'required',
            'amount'   => 'required|array',
            'amount.*' => 'required|numeric|gt:0',
            'username' => [$userValidation],
        ]);

        if (!$id) {
            $user = User::active()->where('username', $request->username)->first();
            if (!$user) {
                $notify[] = ['error', 'User not found'];
                return back()->withNotify($notify);
            }
        }

        if (count($request->name) != count($request->amount) || count($request->name) != count($request->update)) {
            $notify[] = ['error', 'Something went wrong'];
            return back()->withNotify($notify);
        }

        $names       = $request->name;
        $amounts     = $request->amount;
        $updates     = $request->update;
        $totalAmount = array_sum($request->amount);
        $itemCount   = count($request->name);

        if ($id) {
            $invoice       = Invoice::with('invoiceItem')->find($id);
            $notifyMessage = 'Invoice update successfully';
            $oldItems      = $invoice->invoiceItem->pluck('id')->toArray();
            $this->invoiceItemRemove($updates, $oldItems);
        } else {
            $invoice          = new Invoice();
            $invoice->uid     = getTrx();
            $invoice->user_id = $user->id;
            $notifyMessage    = 'Invoice create successfully';
        }

        $invoice->total_item   = $itemCount;
        $invoice->merchant_id  = merchant()->id;
        $invoice->total_amount = $totalAmount;
        $invoice->save();

        $results = array_map(function ($name, $amount, $update) {
            return ['name' => $name, 'price' => $amount, 'update' => $update];
        }, $names, $amounts, $updates);

        foreach ($results as $result) {
            if ($result['update']) {
                $invoice_item = InvoiceItem::find($result['update']);
            } else {
                $invoice_item             = new InvoiceItem();
                $invoice_item->invoice_id = $invoice->id;
            }

            $invoice_item->name  = $result['name'];
            $invoice_item->price = $result['price'];
            $invoice_item->save();
        }

        $notify[] = ['success', $notifyMessage];
        return to_route('merchant.invoice.index')->withNotify($notify);
    }

    private function invoiceItemRemove($updateItems, $newItems) {
        $removeItems = array_diff($newItems, $updateItems);
        foreach ($removeItems as $item) {
            $item = InvoiceItem::find($item);
            $item->delete();
        }
    }

    public function detail($id) {
        $invoice   = Invoice::with(['user', 'merchant'])->where('merchant_id', merchant()->id)->findOrFail($id);
        $pageTitle = 'Invoice Detail - ' . $invoice->uid;
        return view('Template::merchant.invoice.detail', compact('pageTitle', 'invoice'));
    }

    public function cancel(Request $request, $id) {
        $request->validate([
            'cancel_reason' => 'required',
        ]);

        $invoice = Invoice::unpaid()->where('merchant_id', merchant()->id)->find($id);

        if (!$invoice) {
            $notify[] = ['error', 'Invalid invoice'];
            return back()->withNotify($notify);
        }

        $invoice->cancel_reason = $request->cancel_reason;
        $invoice->status        = Status::CANCEL;
        $invoice->save();

        $notify[] = ['success', 'Invoice canceled successfully'];
        return back()->withNotify($notify);
    }

    public function download($id) {
        $pageTitle = 'invoice';
        $invoice   = Invoice::with(['user', 'merchant'])->where('merchant_id', merchant()->id)->findOrFail($id);
        return downloadPDF('invoice.download', compact('invoice', 'pageTitle'));
    }

    public function getUserData(Request $request) {
        $user = User::where('username', $request->username)->first();
        if (!$user) {
            $results = [
                "success" => false,
            ];
            return response()->json($results);
        }

        $results = [
            "success"  => true,
            "fullname" => $user->fullname,
            "mobile"   => $user->mobileNumber,
        ];
        return response()->json($results);
    }
}
