<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Lib\CurlRequest;
use App\Models\AdminNotification;
use App\Models\AutoPayment;
use App\Models\CronJob;
use App\Models\CronJobLog;
use App\Models\MobileOperator;
use App\Models\MobileRecharge;
use App\Models\ModuleSetting;
use App\Models\PointLog;
use App\Models\RedeemReward;
use App\Models\SavingAccount;
use App\Models\SavingInstallment;
use App\Models\SetupUtilityBill;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\User;
use App\Models\UtilityBill;
use Carbon\Carbon;

class CronController extends Controller
{
    public function cron()
    {
        $general            = gs();
        $general->last_cron = now();
        $general->save();

        $crons = CronJob::with('schedule');

        if (request()->alias) {
            $crons->where('alias', request()->alias);
        } else {
            $crons->where('is_running', Status::YES);
        }
        $crons = $crons->get();
        foreach ($crons as $cron) {
            $cronLog              = new CronJobLog();
            $cronLog->cron_job_id = $cron->id;
            $cronLog->start_at    = now();
            if ($cron->is_default) {
                $controller = new $cron->action[0];
                try {
                    $method = $cron->action[1];
                    $controller->$method();
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            } else {
                try {
                    CurlRequest::curlContent($cron->url);
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            }
            $cron->last_run = now();
            $cron->next_run = now()->addSeconds($cron->schedule->interval);
            $cron->save();

            $cronLog->end_at = $cron->last_run;

            $startTime         = Carbon::parse($cronLog->start_at);
            $endTime           = Carbon::parse($cronLog->end_at);
            $diffInSeconds     = $startTime->diffInSeconds($endTime);
            $cronLog->duration = $diffInSeconds;
            $cronLog->save();
        }
        if (request()->target == 'all') {
            $notify[] = ['success', 'Cron executed successfully'];
            return back()->withNotify($notify);
        }
        if (request()->alias) {
            $notify[] = ['success', keyToTitle(request()->alias) . ' executed successfully'];
            return back()->withNotify($notify);
        }
    }

    public function autoPayment()
    {
        $autoPaymentModule = ModuleSetting::active()->where('slug', 'auto_payment')->where('user_type', 'USER')->first();

        if (!$autoPaymentModule) {
            return true;
        }

        $autoPayments = AutoPayment::active()->where('next_payment', showDateTime(now(), 'Y-m-d'))->get();

        foreach ($autoPayments as $autoPayment) {
            $user = $autoPayment->user;

            if (!$user) {
                return false;
            }

            if ($autoPayment->payment_type == 'utility_bill') {
                $this->utilityBill($autoPayment, $user);
            }

            if ($autoPayment->payment_type == 'send_money') {
                $this->sendMoney($autoPayment, $user);
            }

            if ($autoPayment->payment_type == 'mobile_recharge') {
                $this->mobileRecharge($autoPayment, $user);
            }

            $addDay = 0;

            if ($autoPayment->payment_days == 'weekly') {
                $addDay = 7;
            } else if ($autoPayment->payment_days == '15days') {
                $addDay = 15;
            } else if ($autoPayment->payment_days == 'monthly') {
                $addDay = 30;
            }

            $autoPayment->next_payment = showDateTime(now()->addDays($addDay), 'Y-m-d');
            $autoPayment->save();
        }
    }

    private function utilityBill($autoPayment, $user)
    {
        $setupUtility = SetupUtilityBill::active()->autoPayable()->whereHas('form')->find($autoPayment->utility_id);

        if (!$setupUtility) {
            return false;
        }

        $amount        = $autoPayment->amount;
        $fixedCharge   = $setupUtility->fixed_charge;
        $percentCharge = $amount * $setupUtility->percent_charge / 100;
        $totalCharge   = $fixedCharge + $percentCharge;
        $totalAmount   = getAmount($amount + $totalCharge);

        if ($user->balance < $totalAmount) {
            notify($user, 'INSUFFICIENT_BALANCE', [
                'amount'       => showAmount($totalAmount, currencyFormat: false),
                'payment_type' => 'Utility Bill',
                'balance'      => showAmount($user->balance, currencyFormat: false),
            ]);

            return false;
        }

        $user->balance -= $totalAmount;
        $user->save();

        $utilityBill                        = new UtilityBill();
        $utilityBill->user_id               = $user->id;
        $utilityBill->setup_utility_bill_id = $setupUtility->id;
        $utilityBill->amount                = $amount;
        $utilityBill->trx                   = getTrx();
        $utilityBill->user_data             = $autoPayment->user_data;
        $utilityBill->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $amount;
        $transaction->amount        = $totalAmount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = $totalCharge;
        $transaction->charge_type   = '+';
        $transaction->trx_type      = '-';
        $transaction->remark        = 'utility_bill';
        $transaction->details       = 'Utility bill';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = $utilityBill->trx;
        $transaction->save();

        generatePoints($transaction, $user);

        checkUserReward($user, $transaction);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New utility bill request from ' . $user->username;
        $adminNotification->click_url = urlPath('admin.utility.bill.all');
        $adminNotification->save();
    }

    private function sendMoney($autoPayment, $user)
    {
        $receiver = User::where('id', $autoPayment->receiver_id)->first();

        if (!$receiver) {
            return false;
        }

        $sendMoneyCharge = TransactionCharge::where('slug', 'send_money')->first();

        $fixedCharge = $sendMoneyCharge->fixed_charge;
        $totalCharge = ($autoPayment->amount * $sendMoneyCharge->percent_charge / 100) + $fixedCharge;
        $cap         = $sendMoneyCharge->cap;

        if ($sendMoneyCharge->cap != -1 && $totalCharge > $cap) {
            $totalCharge = $cap;
        }

        $totalAmount = getAmount($autoPayment->amount + $totalCharge);
        if ($sendMoneyCharge->daily_limit != -1 && $user->trxLimit('send_money', $user)['daily'] + $totalAmount >= $sendMoneyCharge->daily_limit) {
            notify($user, 'PAYMENT_LIMIT', [
                'amount'       => showAmount($totalAmount, currencyFormat: false),
                'payment_type' => 'Send Money',
                'message'      => 'Daily send money limit has been exceeded',
            ]);

            return false;
        }

        if ($sendMoneyCharge->monthly_limit != 1 && ($user->trxLimit('send_money', $user)['monthly'] + $totalAmount) > $sendMoneyCharge->monthly_limit) {
            notify($user, 'PAYMENT_LIMIT', [
                'amount'       => showAmount($totalAmount, currencyFormat: false),
                'payment_type' => 'Send Money',
                'message'      => 'Your monthly send money limit exceeded',
            ]);
            return false;
        }

        if ($totalAmount > $user->balance) {
            notify($user, 'INSUFFICIENT_BALANCE', [
                'amount'       => showAmount($totalAmount, currencyFormat: false),
                'payment_type' => 'Send Money',
                'balance'      => showAmount($user->balance, currencyFormat: false),
            ]);
            return false;
        }

        $user->balance -= $totalAmount;
        $user->save();

        $senderTrx                = new Transaction();
        $senderTrx->user_id       = $user->id;
        $senderTrx->user_type     = 'USER';
        $senderTrx->before_charge = $autoPayment->amount;
        $senderTrx->amount        = $totalAmount;
        $senderTrx->post_balance  = $user->balance;
        $senderTrx->charge        = $totalCharge;
        $senderTrx->charge_type   = '+';
        $senderTrx->trx_type      = '-';
        $senderTrx->remark        = 'send_money';
        $senderTrx->details       = 'Send Money to';
        $senderTrx->receiver_id   = $receiver->id;
        $senderTrx->receiver_type = "USER";
        $senderTrx->trx           = getTrx();
        $senderTrx->save();

        generatePoints($senderTrx, $user);

        checkUserReward($user, $senderTrx);

        $receiver->balance += $autoPayment->amount;
        $receiver->save();

        $receiverTrx                = new Transaction();
        $receiverTrx->user_id       = $receiver->id;
        $receiverTrx->user_type     = 'USER';
        $receiverTrx->before_charge = $autoPayment->amount;
        $receiverTrx->amount        = $autoPayment->amount;
        $receiverTrx->post_balance  = $receiver->balance;
        $receiverTrx->charge        = 0;
        $receiverTrx->charge_type   = '+';
        $receiverTrx->trx_type      = '+';
        $receiverTrx->remark        = 'receive_money';
        $receiverTrx->details       = 'Received Money From';
        $receiverTrx->receiver_id   = $receiver->id;
        $receiverTrx->receiver_type = "USER";
        $receiverTrx->trx           = $senderTrx->trx;
        $receiverTrx->save();

        notify($user, 'SEND_MONEY', [
            'amount'  => showAmount($totalAmount, currencyFormat: false),
            'charge'  => showAmount($totalCharge, currencyFormat: false),
            'to_user' => $receiver->fullname . ' ( ' . $receiver->username . ' )',
            'trx'     => $senderTrx->trx,
            'time'    => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($user->balance, currencyFormat: false),
        ]);

        notify($receiver, 'RECEIVED_MONEY', [
            'amount'    => showAmount($autoPayment->amount, currencyFormat: false),
            'from_user' => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'       => $senderTrx->trx,
            'time'      => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance'   => showAmount($receiver->balance, currencyFormat: false),
        ]);
    }

    private function mobileRecharge($autoPayment, $user)
    {
        $amount   = $autoPayment->amount;
        $operator = MobileOperator::active()->find($autoPayment->operator_id);
        if (!$operator) {
            notify($user, 'INVALID_MOBILE_OPERATOR', [
                'amount'       => showAmount($amount, currencyFormat: false),
                'payment_type' => 'Mobile Recharge',
                'message'      => 'Operator not found',
            ]);
            return false;
        }

        $mobileRechargeCharge = TransactionCharge::where('slug', 'mobile_recharge')->first();
        if (!$mobileRechargeCharge) {
            return false;
        }

        if ($amount < $mobileRechargeCharge->min_limit || $amount > $mobileRechargeCharge->max_limit) {
            notify($user, 'PAYMENT_LIMIT', [
                'amount'       => showAmount($amount, currencyFormat: false),
                'payment_type' => 'Mobile Recharge',
                'message'      => 'Please Follow the mobile recharge limit',
            ]);
            return false;
        }

        $fixedCharge   = $mobileRechargeCharge->fixed_charge;
        $percentCharge = $amount * $mobileRechargeCharge->percent_charge / 100;
        $totalCharge   = $fixedCharge + $percentCharge;
        $totalAmount   = getAmount($amount + $totalCharge);

        if ($totalAmount > $user->balance) {
            notify($user, 'INSUFFICIENT_BALANCE', [
                'amount'       => showAmount($totalAmount, currencyFormat: false),
                'payment_type' => 'Mobile Recharge',
                'balance'      => showAmount($user->balance, currencyFormat: false),
            ]);
            return false;
        }

        $user->balance -= $totalAmount;
        $user->save();

        $mobileRecharge                     = new MobileRecharge();
        $mobileRecharge->user_id            = $user->id;
        $mobileRecharge->mobile_operator_id = $autoPayment->operator_id;
        $mobileRecharge->mobile             = $autoPayment->user_data->mobile;
        $mobileRecharge->amount             = $amount;
        $mobileRecharge->trx                = getTrx();
        $mobileRecharge->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $amount;
        $transaction->amount        = $totalAmount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = $totalCharge;
        $transaction->charge_type   = '+';
        $transaction->trx_type      = '-';
        $transaction->remark        = 'mobile_recharge';
        $transaction->details       = 'Mobile recharge';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = $mobileRecharge->trx;
        $transaction->save();

        generatePoints($transaction, $user);

        checkUserReward($user, $transaction);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New mobile recharge request from ' . $user->username;
        $adminNotification->click_url = urlPath('admin.mobile.recharge.all');
        $adminNotification->save();
    }

    public function installmentCheck()
    {
        $currentDate = Carbon::now();

        SavingAccount::running()
            ->with('user', 'dpsPlan', 'dpsPlan.savingInterval')
            ->where('next_installment_date', '<=', $currentDate)
            ->chunk(50, function ($accounts) use ($currentDate) {
                foreach ($accounts as $account) {
                    $this->processInstallment($account, $currentDate);
                }
            });
    }

    private function processInstallment($account, $currentDate)
    {
        $nextInstallment = $account->next_installment_date;
        $dpsPlan         = $account->dpsPlan;
        $user            = $account->user;
        $amount          = $dpsPlan->installment_amount;

        $installment                    = new SavingInstallment();
        $installment->user_id           = $user->id;
        $installment->saving_account_id = $account->id;
        $installment->amount            = $amount;
        $installment->trx               = getTrx();

        $penaltyRate       = $account->dpsPlan->penalty_rate  ?? 0;
        $penaltyAmount     = $installment->amount * $penaltyRate / 100;

        if ($user->balance >= $amount) {
            $user->balance -= $amount;
            $user->save();

            $account->balance += $amount;
            $account->last_installment_at = $currentDate;

            $installment->installment_paid_at = $currentDate;
            $installment->status = Status::INSTALLMENT_PAID;
            $installment->save();

            $transaction                = new Transaction();
            $transaction->user_id       = $user->id;
            $transaction->user_type     = 'USER';
            $transaction->amount        = $amount;
            $transaction->post_balance  = $user->balance;
            $transaction->trx_type      = '-';
            $transaction->remark        = 'installment';
            $transaction->details       = 'Installment Payment';
            $transaction->trx           = $installment->trx;
            $transaction->save();

            notify($user, 'SAVINGS_INSTALLMENT_PAID', [
                'trx'             => $transaction->trx,
                'account_id'      => $account->account_id,
                'account_balance' => showAmount($account->balance, currencyFormat: false),
                'amount'          => showAmount($transaction->amount, currencyFormat: false),
                'post_balance'    => showAmount($transaction->post_balance, currencyFormat: false),
                'organization'    => @$dpsPlan->organization?->name,
                'date'            => showDateTime($transaction->created_at),
            ]);
        } else {
            $installment->penalty_amount = $penaltyAmount;
            $installment->status         = Status::INSTALLMENT_UNPAID;
            $installment->save();

            notify($user, 'SAVINGS_INSTALLMENT_DUE', [
                'account_id'      => $account->account_id,
                'amount'          => showAmount($installment->amount, currencyFormat: false),
                'post_balance'    => showAmount($user->balance, currencyFormat: false),
                'organization'    => @$dpsPlan->organization?->name,
                'penalty_amount'  => showAmount($installment->penalty_amount, currencyFormat: false),
                'date'            => showDateTime($installment->created_at, format: 'd M Y'),
            ]);
        }

        $account->next_installment_date = $nextInstallment->copy()->addDays(
            $dpsPlan->savingInterval->installment_interval
        );

        $account->save();

        $this->checkSavingsAccountMaturity($account, $user);
    }

    public function dueInstallmentCheck()
    {
        SavingInstallment::where('status', Status::INSTALLMENT_UNPAID)
            ->withWhereHas('savingAccount', function ($query) {
                $query->where('status', Status::ACCOUNT_RUNNING);
            })->chunk(50, function ($installments) {
                foreach ($installments as $installment) {
                    $this->processDueInstallment($installment);
                }
            });
    }

    private function processDueInstallment($installment)
    {
        $savingAccount      = $installment->savingAccount;
        $user               = $savingAccount->user;
        $penaltyInterval    = $savingAccount->dpsPlan->penaltyInterval->installment_interval;
        $amount             = $installment->amount + $installment->penalty_amount;
        $perPenaltyAmount   = $installment->amount * $savingAccount->dpsPlan->penalty_rate / 100;

        if ($user->balance >= $amount) {

            $user->balance -= $amount;
            $user->save();

            $installment->installment_paid_at = Carbon::now();
            $installment->status = Status::INSTALLMENT_PAID;
            $installment->save();

            $savingAccount->balance += $installment->amount;
            $savingAccount->last_installment_at = Carbon::now();
            $savingAccount->save();

            $transaction                 = new Transaction();
            $transaction->user_id        = $user->id;
            $transaction->user_type      = 'USER';
            $transaction->amount         = $amount;
            $transaction->post_balance   = $user->balance;
            $transaction->trx_type       = '-';
            $transaction->remark         = 'due_installment';
            $transaction->details        = 'Due Installment Payment';
            $transaction->trx            = $installment->trx;
            $transaction->save();

            notify($user, 'SAVINGS_INSTALLMENT_PAID', [
                'trx'             => $transaction->trx,
                'account_id'      => $savingAccount->account_id,
                'account_balance' => showAmount($savingAccount->balance, currencyFormat: false),
                'amount'          => showAmount($transaction->amount, currencyFormat: false),
                'post_balance'    => showAmount($transaction->post_balance, currencyFormat: false),
                'organization'    => @$savingAccount->dpsPlan->organization?->name,
                'date'            => showDateTime($transaction->created_at),
            ]);

            $this->checkSavingsAccountMaturity($savingAccount, $user);
        } else {

            $installmentDate = Carbon::parse($installment->created_at);
            $currentDate     = Carbon::now();
            $daysPast        = $installmentDate->diffInDays($currentDate);

            if ($daysPast >= $penaltyInterval) {
                $installment->penalty_amount = round($daysPast / $penaltyInterval, 2) * $perPenaltyAmount;
                $installment->status = Status::INSTALLMENT_UNPAID;
                $installment->save();
            }

            $newAmount = $installment->amount + $installment->penalty_amount;

            notify($user, 'SAVINGS_INSTALLMENT_DUE', [
                'amount'         => showAmount($newAmount, currencyFormat: false),
                'account_id'      => $savingAccount->account_id,
                'penalty_amount' => showAmount($installment->penalty_amount, currencyFormat: false),
                'post_balance'   => showAmount($user->balance, currencyFormat: false),
                'organization'   => @$savingAccount->dpsPlan->organization?->name,
                'date'           => showDateTime($installment->created_at, format: 'd M Y'),
            ]);
        }
    }

    function checkSavingsAccountMaturity($account, $user)
    {
        $dpsPlan = $account->dpsPlan;

        $accountUnpaidInstallments = $account->installments()->where('status', Status::INSTALLMENT_UNPAID)->exists();

        if ($account->balance >= $account->dpsPlan->investmentAmount() && !$accountUnpaidInstallments) {
            $account->next_installment_date = null;
            $account->status = Status::ACCOUNT_MATURED;
            $account->save();

            notify($user, 'SAVINGS_ACCOUNT_MATURED', [
                'account_id'      => $account->account_id,
                'organization'    => @$dpsPlan->organization?->name,
                'invested_amount' => showAmount($account->balance, currencyFormat: false),
                'maturity_amount' => showAmount($account->dpsPlan->maturity_amount, currencyFormat: false),
                'post_balance'    => showAmount($user->balance, currencyFormat: false),
            ]);

            $user->balance += $account->dpsPlan->maturity_amount;
            $user->save();

            $transaction                 = new Transaction();
            $transaction->user_id        = $user->id;
            $transaction->user_type      = 'USER';
            $transaction->amount         = $account->dpsPlan->maturity_amount;
            $transaction->post_balance   = $user->balance;
            $transaction->trx_type       = '+';
            $transaction->remark         = 'maturity_amount';
            $transaction->details        = 'Savings account maturity amount received';
            $transaction->trx            = getTrx();
            $transaction->save();

            notify($user, 'MATURITY_AMOUNT_RECEIVED', [
                'trx'           => $transaction->trx,
                'account_id'    => $account->account_id,
                'amount'        => showAmount($transaction->amount, currencyFormat: false),
                'post_balance'  => showAmount($transaction->post_balance, currencyFormat: false),
                'organization'  => @$dpsPlan->organization?->name,
            ]);

            // Admin notification
            $adminNotification              = new AdminNotification();
            $adminNotification->user_id     = $user->id;
            $adminNotification->user_type   = 'USER';
            $adminNotification->click_url   = urlPath('admin.saving.account.details', $account->id);
            $adminNotification->title       = 'The account of ' . $user->username . ' has been closed.';
            $adminNotification->save();
        }
    }

    public function pointsValidityCheck()
    {
        PointLog::with("user")
            ->where('expires_at', '<=', Carbon::now())
            ->where('status', Status::POINT_AVAILABLE)
            ->chunk(100, function ($points) {
                foreach ($points as $point) {
                    $point->status = Status::POINT_EXPIRED;
                    $point->save();

                    $user = $point->user;
                    if ($user) {
                        $user->total_points -= $point->points;
                        $user->save();
                    }
                }
            });
    }

    public function redeemRewardsValidityCheck()
    {
        RedeemReward::active()->where('expires_at', '<=', Carbon::now())->chunk(50, function ($redeemReward) {
            foreach ($redeemReward as $reward) {
                $reward->status = Status::REDEEM_EXPIRED;
                $reward->save();
            }
        });
    }
}
