<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\Deposit;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use App\Models\Voucher;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class VoucherController extends Controller
{
    public function userVoucherList()
    {
        $notify[] = "Voucher List";
        $vouchers = Voucher::where('user_type', 'USER')->where('user_id', auth()->id())->orderBy('is_used', "ASC")->apiQuery();
        return responseSuccess('voucher_list', $notify, [
            'vouchers' => $vouchers,
        ]);
    }

    public function userVoucher()
    {
        $notify[]      = "Create Voucher";
        $voucherCharge = TransactionCharge::where('slug', 'voucher_charge')->first();

        return responseSuccess('create_voucher', $notify, [
            'otp_type'       => otpType(),
            'voucher_charge' => $voucherCharge,
            'user'           => auth()->user(),
        ]);
    }

    public function userVoucherCreate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'   => 'required|numeric|gt:0',
            'otp_type' => otpType(validation: true),
            'pin'      => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $voucherCharge = TransactionCharge::where('slug', 'voucher_charge')->first();
        if (!$voucherCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        if ($request->amount < $voucherCharge->min_limit || $request->amount > $voucherCharge->max_limit) {
            $notify[] = 'Please Follow the voucher limit';
            return responseError('validation_error', $notify);
        }

        $myVouchers = Voucher::where('user_type', 'USER')->where('user_id', $user->id)->whereDate('created_at', now())->count();

        if ($voucherCharge->voucher_limit != -1 && $myVouchers >= $voucherCharge->voucher_limit) {
            $notify[] = 'Daily voucher create limit has been exceeded';
            return responseError('validation_error', $notify);
        }

        $fixedCharge = $voucherCharge->fixed_charge;
        $totalCharge = ($request->amount * $voucherCharge->percent_charge / 100) + $fixedCharge;
        $cap         = $voucherCharge->cap;

        if ($voucherCharge->cap != -1 && $totalCharge > $cap) {
            $totalCharge = $cap;
        }

        $totalAmount = $request->amount + $totalCharge;

        if ($totalAmount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction            = new UserActionProcess();
        $userAction->user_id   = auth()->user()->id;
        $userAction->user_type = 'USER';
        $userAction->act       = 'create_voucher';

        $userAction->details = [
            'amount'      => $request->amount,
            'totalAmount' => $totalAmount,
            'totalCharge' => $totalCharge,
            'done_route'  => 'api.voucher.create.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseError('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function userVoucherCreateDone($actionId)
    {
        $userAction = UserAction::where('user_id', auth()->user()->id)->where('user_type', 'USER')->where('is_api', 1)->where('is_used', Status::NO)->where('id', $actionId)->first();
        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;
        $user    = auth()->user();

        if (@$userAction->details->totalAmount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->totalAmount;
        $user->save();

        $voucher               = new Voucher();
        $voucher->user_id      = auth()->id();
        $voucher->user_type    = 'USER';
        $voucher->amount       = $details->amount;
        $voucher->voucher_code = getVoucher();
        $voucher->save();

        $trx                = new Transaction();
        $trx->user_id       = auth()->id();
        $trx->user_type     = 'USER';
        $trx->before_charge = $details->amount;
        $trx->amount        = $details->totalAmount;
        $trx->post_balance  = $user->balance;
        $trx->charge        = $details->totalCharge;
        $trx->charge_type   = '+';
        $trx->trx_type      = '-';
        $trx->remark        = 'create_voucher';
        $trx->details       = 'Voucher created successfully';
        $trx->trx           = getTrx();
        $trx->save();

        generatePoints($trx, $user);

        checkUserReward($user, $trx);

        $notify[] = 'Voucher created successfully';
        return responseSuccess('voucher_create_done', $notify);
    }

    public function userVoucherRedeemLog()
    {
        $notify[] = "Voucher Redeem Log";
        $logs     = Voucher::where('redeemer_id', auth()->id())->where('is_used', Status::YES)->apiQuery();

        return responseSuccess('check_agent', $notify, [
            'logs' => $logs,
        ]);
    }

    public function userVoucherRedeemConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user    = auth()->user();
        $voucher = Voucher::where('voucher_code', $request->code)->where('user_id', '!=', $user->id)->where('is_used', Status::NO)->first();

        if (!$voucher) {
            $notify[] = 'Invalid voucher code or this is one of your voucher';
            return responseError('validation_error', $notify);
        }

        $deposit                  = new Deposit();
        $deposit->user_id         = $user->id;
        $deposit->user_type       = 'USER';
        $deposit->is_web           = $request->is_web ?? 0;
        $deposit->method_code     = 0;
        $deposit->amount          = $voucher->amount;
        $deposit->method_currency = gs('cur_text');
        $deposit->charge          = 0;
        $deposit->rate            = 1;
        $deposit->final_amount    = $voucher->amount;
        $deposit->btc_amount      = 0;
        $deposit->btc_wallet      = "";
        $deposit->trx             = getTrx();
        $deposit->status          = Status::PAYMENT_SUCCESS;
        $deposit->save();

        $user->balance += $voucher->amount;
        $user->save();

        $trx                = new Transaction();
        $trx->user_id       = $user->id;
        $trx->user_type     = 'USER';
        $trx->before_charge = $voucher->amount;
        $trx->amount        = $voucher->amount;
        $trx->post_balance  = $user->balance;
        $trx->charge        = 0;
        $trx->charge_type   = '+';
        $trx->trx_type      = '+';
        $trx->remark        = 'redeem_voucher';
        $trx->details       = 'Redeemed Voucher ';
        $trx->trx           = $deposit->trx;
        $trx->save();

        $voucher->is_used     = Status::YES;
        $voucher->redeemer_id = $user->id;
        $voucher->save();

        $notify[] = getAmount($voucher->amount) . ' ' . $deposit->method_currency . ' has been added to your account';
        return responseSuccess('voucher_redeem_done', $notify);
    }
}
