<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Models\AdminNotification;
use App\Models\DpsPlan;
use App\Models\Form;
use App\Models\Frontend;
use App\Models\Nominee;
use App\Models\Organization;
use App\Models\SavingAccount;
use App\Models\SavingInstallment;
use App\Models\SavingInterval;
use App\Models\SavingTenure;
use App\Models\Transaction;
use App\Rules\FileTypeValidate;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Lib\Api\UserActionProcess;
use App\Models\UserAction;

class SavingAccountController extends Controller
{
    public function data()
    {
        $notify[]                    = "Saving Account Data";
        $data['policies']            = Frontend::where('data_keys', 'savings.policy')->first()->data_values ?? "";
        $data['intervals']           = SavingInterval::active()->orderBy('id', 'asc')->get();
        $data['tenures']             = SavingTenure::active()->orderBy('id', 'asc')->get();
        $data['savings_accounts']    = SavingAccount::where('user_id', auth()->id())->orderBy('id', 'desc')->get();
        $data['nominees']            = Nominee::where('user_id', auth()->id())->orderBy('id', 'desc')->get();
        $data['nominee_image_path']  = getFilePath('verify');
        $data['otp_types']           = otpType();

        $data['amounts']             = DpsPlan::active()
            ->orderBy('installment_amount', 'asc')
            ->get()
            ->unique('installment_amount')
            ->pluck('installment_amount', 'id')
            ->values()
            ->toArray();

        return responseSuccess('saving_account_data', $notify, $data);
    }

    public function list()
    {
        $notify[] = "My Savings Account";
        $query    = SavingAccount::where('user_id', auth()->id())->with('dpsplan', 'dpsplan.organization', 'dpsplan.savingTenure', 'dpsplan.savingInterval', 'installments')->orderBy('id', 'desc');

        $data['accounts']['running']  = (clone $query)->running()->get();
        $data['accounts']['matured']  = (clone $query)->matured()->get();
        $data['accounts']['canceled'] = (clone $query)->cancelled()->get();
        $data['organization_image_path'] = getFilePath('org');

        return responseSuccess('savings_account', $notify, $data);
    }

    public function accountDetails($id)
    {
        $account = SavingAccount::where('id', $id)
            ->where('user_id', auth()->id())
            ->with('dpsplan', 'dpsplan.savingTenure', 'dpsplan.savingInterval', 'dpsplan.organization', 'installments')
            ->first();

        if (!$account) {
            $notify[] = "Sorry! Account not found.";
            return responseError('invalid_request', $notify);
        }

        $notify[] = "Account Details";
        return responseSuccess('account_details', $notify, $account);
    }

    public function accountClosed(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'pin'        => 'required',
        ], [
            'pin.required'        => 'Please enter your account PIN.',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();

        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $account = SavingAccount::running()->where('user_id', $user->id)->find($id);

        if (!$account) {
            $notify[] = "Sorry! Account not found.";
            return responseError('invalid_request', $notify);
        }

        $account->status = Status::ACCOUNT_CLOSED;
        $account->save();

        $user->balance += $account->balance;
        $user->save();

        $transaction                 = new Transaction();
        $transaction->user_id        = $user->id;
        $transaction->user_type      = 'USER';
        $transaction->amount         = $account->balance;
        $transaction->post_balance   = $user->balance;
        $transaction->trx_type       = '+';
        $transaction->remark         = 'savings_account_closed';
        $transaction->details        = 'Savings account closed';
        $transaction->trx            = getTrx();
        $transaction->save();

        notify($user, 'SAVINGS_ACCOUNT_CLOSED', [
            'amount' => showAmount($account->balance, currencyFormat: false),
            'organization' => @$account->dpsPlan->organization?->name,
            'post_balance' => showAmount($user->balance, currencyFormat: false),
        ]);

        $notify[] = "Account closed successfully";
        return responseSuccess('account_closed', $notify, $account);
    }

    public function organizationList(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'saving_tenure_id'   => 'required',
            'saving_interval_id' => 'required',
            'installment_amount' => 'required|exists:dps_plans,installment_amount',
        ]);

        $savingTenure  = SavingTenure::active()->find($request->saving_tenure_id);

        if (!$savingTenure) {
            $notify[] = "Sorry! Saving tenure not found.";
            return responseError('invalid_request', $notify);
        }

        $savingInterval = SavingInterval::active()->find($request->saving_interval_id);

        if (!$savingInterval) {
            $notify[] = "Sorry! Saving interval not found.";
            return responseError('invalid_request', $notify);
        }

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $notify[] = "Organization List";

        $organizations = Organization::active()
            ->withWhereHas('dpsPlans', function ($query) use ($request) {
                $query->where('status', Status::ENABLE)
                    ->where('saving_tenure_id', $request->saving_tenure_id)
                    ->where('saving_interval_id', $request->saving_interval_id)
                    ->where('installment_amount', $request->installment_amount);
            })
            ->orderBy('id', 'asc')
            ->get();

        return responseSuccess('select_organization', $notify, [
            'organizations' => $organizations,
            'organaization_image_path' => getFilePath('org')
        ]);
    }

    public function nomineeForm()
    {
        $notify[] = "Nominee Form";
        $nomineeForm = Form::where('act', 'nominee_form')->first();

        return responseSuccess('nominee_form', $notify, $nomineeForm);
    }

    public function nominees()
    {
        $notify[] = "Nominees";
        $nominees  = auth()->user()->nominees;
        return responseSuccess('nominees', $notify, $nominees);
    }

    public function nomineeCreate(Request $request)
    {
        $nomineeForm = Form::where('act', 'nominee_form')->first();

        if (!$nomineeForm) {
            $notify[] = "Nominee form not found.";
            return responseError('not_found', $notify);
        }

        $user = auth()->user();
        $formData           = $nomineeForm->form_data;
        $formProcessor      = new FormProcessor();
        $formValidationRule = $formProcessor->valueValidation($formData);

        $baseRules = [
            'name'       => 'required|string|max:40',
            'birth_date' => 'required|date_format:Y-m-d',
            'image'      => ['required', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ];

        $rules = array_merge($baseRules, $formValidationRule);

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $nomineeExists = $user->nominees()->where('nominee_name', $request->name)->first();

        if ($nomineeExists) {
            $notify[] = "Your already added this nominee.";
            return responseError('validation_error', $notify);
        }

        $nomineeData = $formProcessor->processFormData($request, $formData);

        $nominee = new Nominee();
        $nominee->user_id            = $user->id;
        $nominee->nominee_name       = $request->name;
        $nominee->nominee_birth_date = $request->birth_date;

        if ($request->hasFile('image')) {
            try {
                $nominee->nominee_image = fileUploader($request->image, getFilePath('verify'), getFileSize('verify'), null);
            } catch (\Exception $e) {
                $notify[] = 'Couldn\'t upload your image';
                return responseError('validation_error', $notify);
            }
        }

        $nominee->nominee_data = $nomineeData;
        $nominee->save();

        $notify[] = "Nominee created successfully.";
        return responseSuccess('nominee', $notify, $nominee);
    }

    public function nomineeDelete(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'nominee_id' => 'required',
            'pin'        => 'required',
        ], [
            'pin.required'        => 'Please enter your account PIN.',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();

        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $nominee = Nominee::where('id', $request->nominee_id)->where('user_id', $user->id)->first();

        if (!$nominee) {
            $notify[] = "Nominee not found.";
            return responseError('not_found', $notify);
        }

        $nomineeAcitveAccontExists = $nominee->savingAccount()->where('status', Status::ACCOUNT_RUNNING)->first();

        if ($nomineeAcitveAccontExists) {
            $notify[] = "Unable to delete, you have an active account with this nominee.";
            return responseError('validation_error', $notify);
        }

        fileManager()->removeFile(getFilePath('verify') . '/' . $nominee->nominee_image);

        foreach (@$nominee->nominee_data ?? [] as $nomineeData) {
            if ($nomineeData->type == 'file') {
                fileManager()->removeFile(getFilePath('verify') . '/' . $nomineeData->value);
            }
        }

        $nominee->delete();

        $notify[] = "Nominee deleted successfully.";
        return responseSuccess('nominee_delete', $notify);
    }

    public function createSavingsAccount(Request $request)
    {
        $rules = [
            'dps_plan_id' => 'required',
            'pin'         => 'required',
            'nominee_id'  => 'required',
            'otp_type' => otpType(validation: true),
        ];

        $rulesMessages = [
            'dps_plan_id.required' => 'Please select a DPS plan.',
            'nominee_id.required'  => 'Please select a nominee.',
            'pin.required'         => 'Please enter your account PIN.',
        ];

        $validator = Validator::make($request->all(), $rules, $rulesMessages);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();

        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $dpsPlan = DpsPlan::active()->find($request->dps_plan_id);

        if (!$dpsPlan) {
            $notify[] = "Sorry, The DPS plan not found";
            return responseError('not_found', $notify);
        }

        $nominee = Nominee::where('user_id', $user->id)
            ->where('id', $request->nominee_id)
            ->first();

        if (!$nominee) {
            $notify[] = "Sorry, The nominee not found";
            return responseError('not_found', $notify);
        }


        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'savings_account';

        $userAction->details = [
            'user_id'          => $user->id,
            'dps_plan_id'      => $dpsPlan->id,
            'nominee_id'       => $nominee->id,
            'done_route'       => 'api.savings-account.create.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }
        return callApiMethod($userAction->next_route, $actionId);
    }

    public function createSavingsAccountDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;

        $userAction->is_used = Status::YES;
        $userAction->save();

        $dpsPlan = DpsPlan::active()->find($details->dps_plan_id);

        if (!$dpsPlan) {
            $notify[] = "Sorry, The DPS plan not found";
            return responseError('not_found', $notify);
        }

        // Create savings account
        $savingsAccount                         = new SavingAccount();
        $savingsAccount->account_id             = getNumber();
        $savingsAccount->user_id                = $user->id;
        $savingsAccount->dps_plan_id            = $dpsPlan->id;
        $savingsAccount->organization_id        = $dpsPlan->organization_id;
        $savingsAccount->nominee_id             = $details->nominee_id;
        $savingsAccount->next_installment_date  = Carbon::now()->addDays($dpsPlan->savingInterval->installment_interval);
        $savingsAccount->save();

        // first installment
        $installment                     = new SavingInstallment();
        $installment->trx                = getTrx();
        $installment->amount             = $dpsPlan->installment_amount;
        $installment->user_id            = $user->id;
        $installment->saving_account_id  = $savingsAccount->id;

        if ($user->balance >= $dpsPlan->installment_amount) {
            $user->balance -= $dpsPlan->installment_amount;
            $user->save();

            $installment->installment_paid_at  = Carbon::now();
            $installment->status = Status::INSTALLMENT_PAID;
            $installment->save();

            // transaction
            $transaction                = new Transaction();
            $transaction->user_id       = $user->id;
            $transaction->user_type     = 'USER';
            $transaction->amount        = $installment->amount;
            $transaction->post_balance  = $user->balance;
            $transaction->trx           = $installment->trx;
            $transaction->trx_type      = '-';
            $transaction->remark        = 'installment';
            $transaction->details       = 'First installment to ' . @$dpsPlan->organization?->name;
            $transaction->save();

            // update account balance
            $savingsAccount->balance = $dpsPlan->installment_amount;
            $savingsAccount->save();

            notify($user, 'SAVINGS_INSTALLMENT_PAID', [
                'trx'          => $installment->trx,
                'amount'       => showAmount($dpsPlan->installment_amount),
                'date'         => showDateTime(Carbon::now()),
                'post_balance' => showAmount($user->balance),
            ]);
        } else {
            $penaltyRate       = $savingsAccount->dpsPlan->penalty_rate  ?? 0;
            $penaltyAmount     = ($savingsAccount->dpsPlan->installment_amount * $penaltyRate) / 100;

            $installment->status = Status::INSTALLMENT_UNPAID;
            $installment->penalty_amount = $penaltyAmount;
            $installment->save();

            notify($user, 'SAVINGS_INSTALLMENT_DUE', [
                'amount'         => showAmount($installment->amount, currencyFormat: false),
                'post_balance'   => showAmount($user->balance, currencyFormat: false),
                'organization'   => @$dpsPlan->organization?->name,
                'penalty_amount' => showAmount($penaltyAmount, currencyFormat: false),
                'date'           => showDateTime($installment->created_at, format: 'd M Y'),
            ]);
        }

        // Admin notification
        $adminNotification              = new AdminNotification();
        $adminNotification->user_id     = $user->id;
        $adminNotification->user_type   = 'USER';
        $adminNotification->click_url   = urlPath('admin.saving.account.details', $savingsAccount->id);
        $adminNotification->title       = $user->username . ' created a saving account.';
        $adminNotification->save();

        notify($user, 'SAVINGS_ACCOUNT_CREATED', [
            'amount'           => showAmount($installment->amount, currencyFormat: false),
            'installment_date' => showDateTime($savingsAccount->next_installment_date),
            'organization'     => @$dpsPlan->organization?->name,
        ]);

        $notify[] = "Savings account created.";
        return responseSuccess('saving_account', $notify);
    }
}
