<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Lib\FormProcessor;
use App\Models\AdminNotification;
use App\Models\SetupUtilityBill;
use App\Models\Transaction;
use App\Models\UserAction;
use App\Models\UtilityBill;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Barryvdh\DomPDF\Facade\Pdf;

class PayBillController extends Controller
{
    public function payBill()
    {
        $notify[] = "Pay Bill";

        $user    = auth()->user();
        $utility = SetupUtilityBill::active()->with('form')->whereHas('form')->get();
        return responseSuccess('pay_bill', $notify, [
            'otp_type'                => otpType(),
            'current_balance'         => $user->balance,
            'utility'                 => $utility,
            'latest_pay_bill_history' => UtilityBill::where('user_id', $user->id)->with('setupUtilityBill', 'getTrx')->groupBy('setup_utility_bill_id')->orderBy('id', 'DESC')->limit(10)->get(),
        ]);
    }

    public function payBillConfirm(Request $request)
    {
        $utility = SetupUtilityBill::active()->whereHas('form')->find($request->utility_id);
        if (!$utility) {
            $notify[] = 'Sorry, Utility bill not found';
            return responseError('validation_error', $notify);
        }

        $formData       = $utility->form->form_data;
        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $userData       = $formProcessor->processFormData($request, $formData);

        $validator = Validator::make($request->all(), array_merge([
            'amount'     => 'required|gt:0',
            'utility_id' => 'required',
            'otp_type'   => otpType(validation: true),
            'pin'        => 'required',
        ], $validationRule));

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $fixedCharge   = $utility->fixed_charge;
        $percentCharge = $request->amount * $utility->percent_charge / 100;
        $totalCharge   = $fixedCharge + $percentCharge;
        $totalAmount   = getAmount($request->amount + $totalCharge);

        if ($totalAmount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'utility_bill';

        $userAction->details = [
            'amount'       => $request->amount,
            'total_amount' => $totalAmount,
            'total_charge' => $totalCharge,
            'utility_id'   => $utility->id,
            'user_data'    => $userData,
            'done_route'   => 'api.pay.bill.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function payBillDone($actionId)
    {

        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;
        if (@$details->total_amount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $setupUtility = SetupUtilityBill::find($details->utility_id);

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->total_amount;
        $user->save();

        $utilityBill                        = new UtilityBill();
        $utilityBill->user_id               = $user->id;
        $utilityBill->setup_utility_bill_id = $setupUtility->id;
        $utilityBill->amount                = $details->amount;
        $utilityBill->trx                   = getTrx();
        $utilityBill->user_data             = $details->user_data;
        $utilityBill->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $details->amount;
        $transaction->amount        = $details->total_amount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = $details->total_charge;
        $transaction->charge_type   = '+';
        $transaction->trx_type      = '-';
        $transaction->remark        = 'utility_bill';
        $transaction->details       = 'Utility bill';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = $utilityBill->trx;
        $transaction->save();

        generatePoints($transaction, $user);

        checkUserReward($user, $transaction);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New utility bill request from ' . $user->username;
        $adminNotification->click_url = urlPath('admin.utility.bill.all');
        $adminNotification->save();

        $notify[] = 'Successfully complete the utility bill process';
        return responseSuccess('utility_bill_done', $notify, [
            'utility_bill' => $utilityBill->load('setupUtilityBill'),
            'transaction'  => $transaction,
        ]);
    }

    public function payBillHistory()
    {
        $notify[] = "Pay Bill History";
        return responseSuccess('pay_bill_history', $notify, [
            'file_path' => route('home') . '/' . getFilePath('verify'),
            'history'   => UtilityBill::where('user_id', auth()->id())->with('setupUtilityBill', 'getTrx')->apiQuery(),
        ]);
    }

    public function payBillDownload($id)
    {
        $payBill = UtilityBill::where('user_id', auth()->id())->with('user', 'setupUtilityBill')->find($id);
        if (!$payBill) {
            $notify[] = 'Bill not found';
            return responseError('bill_not_found', $notify);
        }
        $pdf = Pdf::loadView('templates.basic.user.pdf', compact('payBill'));
        return $pdf->download('paybill.pdf');
    }
}
