<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\AdminNotification;
use App\Models\MobileOperator;
use App\Models\MobileRecharge;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class MobileRechargeController extends Controller
{
    public function mobileRecharge()
    {
        $notify[] = "Mobile Recharge";

        $user                 = auth()->user();
        $mobileRechargeCharge = TransactionCharge::where('slug', 'mobile_recharge')->first();
        $mobileOperators      = MobileOperator::active()->get();
        return responseSuccess('money_recharge', $notify, [
            'otp_type'                       => otpType(),
            'current_balance'                => $user->balance,
            'mobile_recharge_charge'         => $mobileRechargeCharge,
            'mobile_operators'               => $mobileOperators,
            'latest_mobile_recharge_history' => MobileRecharge::where('user_id', $user->id)->with('mobileOperator')->groupBy('mobile')->orderBy('id', 'DESC')->limit(10)->get(),
        ]);
    }

    public function mobileRechargeConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'      => 'required|gt:0',
            'mobile'      => 'required',
            'operator_id' => 'required',
            'otp_type'    => otpType(validation: true),
            'pin'         => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $operator = MobileOperator::active()->find($request->operator_id);
        if (!$operator) {
            $notify[] = 'Sorry, Operator not found';
            return responseError('validation_error', $notify);
        }

        $mobileRechargeCharge = TransactionCharge::where('slug', 'mobile_recharge')->first();
        if (!$mobileRechargeCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        if ($request->amount < $mobileRechargeCharge->min_limit || $request->amount > $mobileRechargeCharge->max_limit) {
            $notify[] = 'Please Follow the mobile recharge limit';
            return responseError('validation_error', $notify);
        }

        $fixedCharge   = $mobileRechargeCharge->fixed_charge;
        $percentCharge = $request->amount * $mobileRechargeCharge->percent_charge / 100;
        $totalCharge   = $fixedCharge + $percentCharge;
        $totalAmount   = getAmount($request->amount + $totalCharge);

        if ($totalAmount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'mobile_recharge';

        $userAction->details = [
            'amount'       => $request->amount,
            'total_amount' => $totalAmount,
            'total_charge' => $totalCharge,
            'operator_id'  => $operator->id,
            'mobile'       => $request->mobile,
            'done_route'   => 'api.mobile.recharge.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function mobileRechargeDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;
        if (@$details->total_amount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $operator = MobileOperator::find($details->operator_id);

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->total_amount;
        $user->save();

        $mobileRecharge                     = new MobileRecharge();
        $mobileRecharge->user_id            = $user->id;
        $mobileRecharge->mobile_operator_id = $operator->id;
        $mobileRecharge->mobile             = $details->mobile;
        $mobileRecharge->amount             = $details->amount;
        $mobileRecharge->trx                = getTrx();
        $mobileRecharge->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $details->amount;
        $transaction->amount        = $details->total_amount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = $details->total_charge;
        $transaction->charge_type   = '+';
        $transaction->trx_type      = '-';
        $transaction->remark        = 'mobile_recharge';
        $transaction->details       = 'Mobile recharge';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = $mobileRecharge->trx;
        $transaction->save();

        generatePoints($transaction, $user);

        // check for reward
        checkUserReward($user, $transaction);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New mobile recharge request from ' . $user->username;
        $adminNotification->click_url = urlPath('admin.mobile.recharge.all');
        $adminNotification->save();
        
        $notify[] = 'Successfully complete the mobile recharge process';
        return responseSuccess('mobile_recharge_done', $notify, [
            'mobile_recharge' => $mobileRecharge->load('mobileOperator'),
            'transaction'     => $transaction,
        ]);
    }

    public function mobileRechargeHistory()
    {
        $notify[] = "Mobile Recharge History";
        return responseSuccess('mobile_recharge_history', $notify, [
            'history' => MobileRecharge::where('user_id', auth()->id())->with('mobileOperator', 'getTrx')->apiQuery(),
        ]);
    }
}
