<?php

namespace App\Http\Controllers\Api\Merchant;

use Carbon\Carbon;
use App\Models\Form;
use App\Models\User;
use App\Models\QrCode;
use App\Traits\Common;
use GuzzleHttp\Client;
use App\Constants\Status;
use App\Lib\FormProcessor;
use App\Models\Withdrawal;
use App\Models\DeviceToken;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use App\Rules\FileTypeValidate;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Intervention\Image\ImageManager;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Intervention\Image\Drivers\Gd\Driver;

class MerchantController extends Controller
{
    use Common;

    public function dashboard()
    {
        $user = auth()->user('merchant');

        $totalWithdraw = Withdrawal::where('user_type', 'MERCHANT')
            ->where('user_id', $user->id)
            ->where('withdrawals.status', Status::PAYMENT_SUCCESS)
            ->selectRaw('SUM(amount) as finalAmount')->first();

        $latestTrx = Transaction::where('user_id', $user->id)->where('user_type', 'MERCHANT')
            ->with('receiverUser', 'receiverAgent', 'receiverMerchant')->orderBy('id', 'desc')->take(10)
            ->get();

        $date    = Carbon::today()->subDays(7);
        $moneyIn = Transaction::where('user_id', $user->id)->where('user_type', 'MERCHANT')->where('trx_type', '+')->whereDate('created_at', '>=', $date)
            ->get(['amount']);

        $moneyOut = Transaction::where('user_id', $user->id)->where('user_type', 'MERCHANT')->where('trx_type', '-')->whereDate('created_at', '>=', $date)
            ->get(['amount']);

        $totalMoneyIn  = 0;
        $totalMoneyOut = 0;

        $in = [];
        foreach ($moneyIn as $inTrx) {
            $in[] = $inTrx->amount;
        }
        $totalMoneyIn = array_sum($in);

        $out = [];
        foreach ($moneyOut as $outTrx) {
            $out[] = $outTrx->amount;
        }

        $totalMoneyOut   = array_sum($out);
        $totalMoneyInOut = ['totalMoneyIn' => $totalMoneyIn, 'totalMoneyOut' => $totalMoneyOut];

        $general  = GeneralSetting::first();
        $notify[] = 'Dashboard';
        return responseSuccess('dashboard',$notify,[
            'merchant'                => $user,
            'latest_trx'              => $latestTrx,
            'last_7_day_money_in_out' => $totalMoneyInOut,
            'total_withdraw'          => showAmount($totalWithdraw->finalAmount ?? 0, $general->currency),
        ]);
    }

    public function userInfo()
    {
        $notify[] = 'User information';
        return responseSuccess('user_info',$notify, [
            'user' => auth()->user('merchant'),
        ]);
    }

    public function userDataSubmit(Request $request)
    {
        $user = auth()->user('merchant');
        if ($user->profile_complete == Status::YES) {
            $notify[] = 'You\'ve already completed your profile';
            return responseError('already_completed',$notify);
        }
        $validator = Validator::make($request->all(), [
            'firstname' => 'required',
            'lastname'  => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $user->firstname = $request->firstname;
        $user->lastname  = $request->lastname;
        $user->address   = [
            'country' => @$user->address->country,
            'address' => $request->address,
            'state'   => $request->state,
            'zip'     => $request->zip,
            'city'    => $request->city,
        ];
        $user->profile_complete = Status::YES;
        $user->save();

        $notify[] = 'Profile completed successfully';
        return responseSuccess('profile_completed',$notify);
    }

    public function kycForm()
    {
        if (auth()->user('merchant')->kv == 2) {
            $notify[] = 'Your KYC is under review';
            return responseError('under_review',$notify,[
                'kyc_data' => auth()->user()->kyc_data,
                'path'     => getFilePath('verify'),
            ]);
        }
        if (auth()->user('merchant')->kv == 1) {
            $notify[] = 'You are already KYC verified';
            return responseError('already_verified',$notify);
        }
        $form     = Form::where('act', 'merchant_kyc')->first();
        $notify[] = 'KYC field is below';
        return responseSuccess('kyc_form',$notify,[
            'form' => $form->form_data,
        ]);
    }

    public function kycSubmit(Request $request)
    {
        $form           = Form::where('act', 'merchant_kyc')->first();
        $formData       = $form->form_data;
        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);

        $validator = Validator::make($request->all(), $validationRule);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $userData       = $formProcessor->processFormData($request, $formData);
        $user           = auth()->user('merchant');
        $user->kyc_data = $userData;
        $user->kv       = Status::KYC_PENDING;
        $user->save();

        $notify[] = 'KYC data submitted successfully';
        return responseSuccess('kyc_submitted',$notify);
    }

    public function transactions(Request $request)
    {
        $times        = ['7days', '15days', '30days', '365days'];
        $remarks      = Transaction::where('user_type', 'MERCHANT')->distinct('remark')->orderBy('remark')->pluck('remark')->toArray();
        $transactions = Transaction::where('user_type', 'MERCHANT')->where('user_id', auth()->id());

        if ($request->search) {
            $transactions = $transactions->where('trx', $request->search);
        }

        if ($request->type) {
            $type         = $request->type == 'plus' ? '+' : '-';
            $transactions = $transactions->where('trx_type', $type);
        }

        if ($request->remark) {
            $transactions = $transactions->where('remark', $request->remark);
        }

        if ($request->days) {
            $transactions = $transactions->where('created_at', '>=', now()->subDays($request->days));
        }

        $transactions = $transactions->with(['receiverUser', 'receiverAgent', 'receiverMerchant'])->orderBy('id', 'desc')->apiQuery();

        $notify[] = 'Transactions data';
        return responseSuccess('transactions',$notify,[
            'transactions' => $transactions,
            'remarks'      => $remarks,
            'times'        => $times,
        ]);
    }

    public function apiKey()
    {
        $merchant = auth()->user('merchant');

        if (!$merchant->public_api_key || !$merchant->secret_api_key) {
            $merchant->public_api_key = keyGenerator();
            $merchant->secret_api_key = keyGenerator();
            $merchant->save();
        }

        $notify[] = 'Business Api Key';
        return responseSuccess('business_api_key',$notify,[
            'merchant' => $merchant,
        ]);
    }

    public function generateApiKey()
    {
        $merchant  = auth()->user('merchant');
        $publicKey = keyGenerator();
        $secretKey = keyGenerator();

        $merchant->public_api_key = $publicKey;
        $merchant->secret_api_key = $secretKey;
        $merchant->save();

        $notify[] = 'New API key generated successfully';
        return responseSuccess('key_generated',$notify,[
            'merchant' => $merchant,
        ]);
    }

    public function submitProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'firstname' => 'required',
            'lastname'  => 'required',
            'image'     => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $user = auth()->user('merchant');

        $user->firstname = $request->firstname;
        $user->lastname  = $request->lastname;
        $user->address   = [
            'country' => @$user->address->country,
            'address' => $request->address,
            'state'   => $request->state,
            'zip'     => $request->zip,
            'city'    => $request->city,
        ];

        if ($request->hasFile('image')) {
            try {
                $old         = $user->image;
                $user->image = fileUploader($request->image, getFilePath('merchantProfile'), getFileSize('merchantProfile'), $old);
            } catch (\Exception $exp) {
                $notify[] = 'Couldn\'t upload your image';
                return responseError('validation_error',$notify);
            }
        }

        $user->save();

        $notify[] = 'Profile updated successfully';
        return responseSuccess('profile_updated',$notify);
    }

    public function submitPassword(Request $request)
    {
        $passwordValidation = Password::min(6);
        $general            = gs();
        if ($general->secure_password) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'password'         => ['required', 'confirmed', $passwordValidation],
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $user = auth()->user('merchant');
        if (Hash::check($request->current_password, $user->password)) {
            $password       = Hash::make($request->password);
            $user->password = $password;
            $user->save();
            $notify[] = 'Password changed successfully';
            return responseSuccess('password_changed',$notify);
        } else {
            $notify[] = 'The password doesn\'t match!';
            return responseError('validation_error',$notify);
        }
    }

    public function qrCode()
    {
        $notify[] = 'QR Code';
        $user     = auth()->user('merchant');
        $qrCode   = $user->qrCode;

        if (!$qrCode) {
            $qrCode              = new QrCode();
            $qrCode->user_id     = $user->id;
            $qrCode->user_type   = 'MERCHANT';
            $qrCode->unique_code = keyGenerator(15);
            $qrCode->save();
        }

        $uniqueCode = $qrCode->unique_code;
        $qrCode     = cryptoQR($uniqueCode);
        return responseSuccess('qr_code',$notify, [
            'qr_code' => $qrCode,
        ]);
    }

    public function qrCodeDownload()
    {
        $user    = auth()->user('merchant');
        $qrCode  = $user->qrCode()->first();
        $general = gs();

        $file     = cryptoQR($qrCode->unique_code);
        $filename = $qrCode->unique_code . '.jpg';

        $manager  = new ImageManager(new Driver());
        $template = $manager->read('assets/images/qr_code_template/' . $general->qr_code_template);

        $client       = new Client();
        $response     = $client->get($file);
        $imageContent = $response->getBody()->getContents();

        $qrCode = $manager->read($imageContent)->cover(2000, 2000);
        $template->place($qrCode, 'center');
        $image = $template->encode();

        $headers = [
            'Content-Type'        => 'image/jpeg',
            'Content-Disposition' => 'attachment; filename=' . $filename,
        ];

        return response()->stream(function () use ($image) {
            echo $image;
        }, 200, $headers);
    }

    public function qrCodeRemove(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file_name' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $file = getFilePath('temporary') . '/' . $request->file_name;

        if (file_exists($file)) {
            unlink($file);
            $notify[] = 'QR code removed successfully';
            return responseSuccess('qr_code_remove',$notify);
        }
        $notify[] = 'Already removed';
        return responseSuccess('qr_code_remove',$notify);
    }

    public function getDeviceToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $deviceToken = DeviceToken::where('token', $request->token)->where('user_type', 'MERCHANT')->first();

        if ($deviceToken) {
            $notify[] = 'Already exists';
            return responseSuccess('get_device_token',$notify);
        }

        $deviceToken            = new DeviceToken();
        $deviceToken->user_id   = auth()->user('merchant')->id;
        $deviceToken->user_type = 'MERCHANT';
        $deviceToken->token     = $request->token;
        $deviceToken->is_app    = Status::YES;
        $deviceToken->save();

        $notify[] = 'Token save successfully';
        return responseSuccess('get_device_token',$notify);
    }

    public function show2faData()
    {
        $general   = gs();
        $ga        = new GoogleAuthenticator();
        $user      = auth()->user();
        $secret    = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . $general->site_name, $secret);

        $notify[] = 'Two factor';
        return responseSuccess('remark',$notify,[
            'qr_code_url' => $qrCodeUrl,
            'secret'      => $secret,
        ]);
    }

    public function create2fa(Request $request)
    {
        $user      = auth()->user();
        $validator = Validator::make($request->all(), [
            'key'  => 'required',
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts  = 1;
            $user->save();

            $notify[] = 'Google authenticator activated successfully';
            return responseSuccess('2fa_activated',$notify);
        } else {
            $notify[] = 'Wrong verification code';
            return responseError('wrong_code',$notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $user     = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts  = 0;
            $user->save();

            $notify[] = 'Two factor authenticator deactivated successfully';
            return responseSuccess('2fa_deactivated',$notify);
        } else {
            $notify[] = 'Wrong verification code';
            return responseError('wrong_code',$notify);
        }
    }

    public function accountDelete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $user = auth()->user('merchant');

        if (!Hash::check($request->password, $user->password)) {
            $notify[] = 'Invalid Password';
            return responseError('invalid_password',$notify);
        }

        $user->is_deleted = 1;
        $user->save();

        $user->tokens()->delete();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->user_type = 'MERCHANT';
        $adminNotification->title     = $user->username . ' deleted his account.';
        $adminNotification->click_url = urlPath('admin.merchants.detail', $user->id);
        $adminNotification->save();

        $notify[] = 'Account deleted successfully';
        return responseSuccess('account_deleted',$notify);
    }


    public function getUserData(Request $request)
    {
        $user = User::where('username', $request->username)->orWhere('mobile', $request->username)->first();

        if (!$user) {
            $notify[]= 'User not found';
            return responseError('validation_error',$notify);
        }

        $notify[] = 'Check User';
        return responseSuccess('check_user',$notify,[
            'user' => $user,
        ]);
    }
}
