<?php

namespace App\Http\Controllers\Api\Merchant;

use App\Models\User;
use App\Models\Invoice;
use App\Constants\Status;
use App\Models\InvoiceItem;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\TransactionCharge;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices  = Invoice::searchable(['uid', 'user:username', 'user:mobile', 'user:email'])->with(['user:id,username,firstname,lastname,email,dial_code,mobile', 'merchant:id,username,firstname,lastname,email,mobile', 'invoiceItem'])->where('merchant_id', merchant()->id)->latest()->paginate(getPaginate());
        $paymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        $notify[] = 'Invoice All';
        return responseSuccess('invoice_all',$notify,[
            'current_balance'           => merchant()->balance,
            'payment_charge'         => $paymentCharge,
            'invoices' => $invoices,
        ]);
    }

    public function edit($id)
    {
        $invoice   = Invoice::unpaid()->withCount('invoiceItem')->with(['invoiceItem', 'user:id,username,email,mobile', 'merchant:id,username,email,mobile'])->where('merchant_id', merchant()->id)->find($id);

        if (!$invoice) {
            $notify[] = 'Invoice not found';
            return responseError('not_found',$notify);
        }

        $pageTitle = "Edit Invoice - " . $invoice->uid;
        return responseSuccess('edit_invoice',$pageTitle,[
            'invoice' => $invoice,
        ]);
    }

    public function store(Request $request, $id = null)
    {
        $userValidation = $id ? 'nullable' : 'required';

        $validator = Validator::make($request->all(), [
            'name'     => 'required|array',
            'name.*'   => 'required',
            'amount'   => 'required|array',
            'amount.*' => 'required|numeric|gt:0',
            'username' => [$userValidation]
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        if (!$id) {
            $user = User::active()->where('username', $request->username)->first();
            if (!$user) {
                $notify[] = 'User not found';
                return responseError('not_found',$notify);
            }
        }

        if (count($request->name) != count($request->amount)) {
            $notify[] = 'Something went wrong';
            return responseError('something_wrong',$notify);
        }

        $names       = $request->name;
        $amounts     = $request->amount;
        $totalAmount = array_sum($request->amount);
        $itemCount   = count($request->name);

        if ($id) {
            $invoice       = Invoice::with('invoiceItem')->find($id);
            $notifyMessage = 'Invoice update successfully';
        } else {
            $invoice          = new Invoice();
            $invoice->uid     = getTrx();
            $invoice->user_id = $user->id;
            $notifyMessage    = 'Invoice create successfully';
        }

        $invoice->total_item   = $itemCount;
        $invoice->merchant_id  = merchant()->id;
        $invoice->total_amount = $totalAmount;
        $invoice->save();

        InvoiceItem::where('invoice_id', $invoice->id)->delete();

        $results = array_map(function ($name, $amount) {
            return ['name' => $name, 'price' => $amount];
        }, $names, $amounts);

        foreach ($results as $result) {
            $invoice_item             = new InvoiceItem();
            $invoice_item->invoice_id = $invoice->id;
            $invoice_item->name       = $result['name'];
            $invoice_item->price      = $result['price'];
            $invoice_item->save();
        }

        $invoice = Invoice::with('invoiceItem')->find($invoice->id);
        $notify[] = $notifyMessage;
        return responseSuccess('create_invoice',$notify,[
            'invoice' => $invoice,
        ]);
    }

    public function cancel(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'cancel_reason' => 'required'
        ]);

        if ($validator->fails()) {
            return responseError('validation_error',$validator->errors());
        }

        $invoice   = Invoice::unpaid()->where('merchant_id', merchant()->id)->find($id);

        if (!$invoice) {
            $notify[] = 'Invalid invoice';
            return responseError('not_found',$notify);
        }

        $invoice->cancel_reason = $request->cancel_reason;
        $invoice->status = Status::CANCELED;
        $invoice->save();
        $notify[] = 'Invoice canceled successfully';
        return responseSuccess('cancel_invoice',$notify,[
            'invoice' => $invoice,
        ]);
    }

    public function download($id)
    {
        $pageTitle = 'Download Invoice';
        $invoice = Invoice::with(['user', 'merchant'])->where('merchant_id', merchant()->id)->find($id);

        if (!$invoice) {
            $notify[] = 'Invoice not found';  
            return responseError('not_found',$notify);
        }

        $pdf = Pdf::loadView('invoice.download', compact('invoice', 'pageTitle'));

        return response()->streamDownload(
            fn() => print($pdf->output()),
            'invoice_' . $invoice->uid . '.pdf',
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="invoice_' . $invoice->uid . '.pdf"',
            ]
        );
    }
}
