<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\Merchant;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class MakePaymentController extends Controller
{
    public function checkUser(Request $request)
    {
        $merchant = Merchant::where('username', $request->merchant)->orWhere('mobile', $request->merchant)->first();
        if (!$merchant) {
            $notify[] = 'Merchant out found';
            return responseError('validation_error', $notify);
        }
        $notify[] = 'Check Merchant';
        return responseSuccess('check_merchant', $notify, [
            'merchant' => $merchant,
        ]);
    }

    public function makePayment()
    {
        $notify[] = "Make Payment";

        $user              = auth()->user();
        $makePaymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();

        return responseSuccess('make_payment', $notify, [
            'otp_type'                    => otpType(),
            'current_balance'             => $user->balance,
            'make_payment_charge'         => $makePaymentCharge,
            'latest_make_payment_history' => Transaction::where('user_id', $user->id)
                ->where('remark', 'make_payment')
                ->where('user_type', 'USER')
                ->orderBy('id', 'DESC')
                ->groupBy('receiver_id')
                ->with('receiverMerchant')
                ->limit(10)
                ->get(),
        ]);
    }

    public function makePaymentConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'    => 'required|gt:0',
            'reference' => 'nullable',
            'merchant'  => 'required',
            'otp_type'  => otpType(validation: true),
            'pin'       => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        //Find merchant
        $merchant = Merchant::where('username', $request->merchant)->orWhere('mobile', $request->merchant)->first();
        if (!$merchant) {
            $notify[] = 'Merchant not found';
            return responseError('validation_error', $notify);
        }

        $makePaymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        if (!$makePaymentCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        $merchantFixedCharge   = $makePaymentCharge->merchant_payment_fixed_charge;
        $merchantPercentCharge = $request->amount * $makePaymentCharge->merchant_payment_percent_charge / 100;
        $merchantTotalCharge   = $merchantFixedCharge + $merchantPercentCharge;

        $userFixedCharge   = $makePaymentCharge->user_payment_fixed_charge;
        $userPercentCharge = $request->amount * $makePaymentCharge->user_payment_percent_charge / 100;
        $userTotalCharge   = $userFixedCharge + $userPercentCharge;

        $merchantTotalCharge = getAmount($merchantTotalCharge);
        $merchantTotalAmount = getAmount($request->amount - $merchantTotalCharge);

        $userTotalCharge = getAmount($userTotalCharge);
        $userTotalAmount = getAmount($request->amount + $userTotalCharge);

        if ($userTotalAmount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'make_payment';

        $userAction->details = [
            'amount'                => $request->amount,
            'merchant_total_amount' => $merchantTotalAmount,
            'merchant_total_charge' => $merchantTotalCharge,
            'user_total_amount'     => $userTotalAmount,
            'user_total_charge'     => $userTotalCharge,
            'merchant_id'           => $merchant->id,
            'done_route'            => 'api.make.payment.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseError('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function makePaymentDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;

        $merchant = Merchant::where('id', $details->merchant_id)->first();
        if (!$merchant) {
            $notify[] = 'Sorry! Merchant not found';
            return responseError('validation_error', $notify);
        }

        if (@$details->user_total_amount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->user_total_amount;
        $user->save();

        $senderTrx                = new Transaction();
        $senderTrx->user_id       = $user->id;
        $senderTrx->user_type     = 'USER';
        $senderTrx->before_charge = $details->amount;
        $senderTrx->amount        = $details->user_total_amount;
        $senderTrx->post_balance  = $user->balance;
        $senderTrx->charge        = $details->user_total_charge;
        $senderTrx->charge_type   = '+';
        $senderTrx->trx_type      = '-';
        $senderTrx->remark        = 'make_payment';
        $senderTrx->details       = 'Make payment';
        $senderTrx->receiver_id   = $merchant->id;
        $senderTrx->receiver_type = "MERCHANT";
        $senderTrx->trx           = getTrx();
        $senderTrx->save();

        generatePoints($senderTrx, $user);

        checkUserReward($user, $senderTrx);

        $merchant->balance += $details->merchant_total_amount;
        $merchant->save();

        $merchantTrx                = new Transaction();
        $merchantTrx->user_id       = $merchant->id;
        $merchantTrx->user_type     = 'MERCHANT';
        $merchantTrx->before_charge = $details->amount;
        $merchantTrx->amount        = $details->merchant_total_amount;
        $merchantTrx->post_balance  = $merchant->balance;
        $merchantTrx->charge        = $details->merchant_total_charge;
        $merchantTrx->charge_type   = '-';
        $merchantTrx->trx_type      = '+';
        $merchantTrx->remark        = 'payment_receive';
        $merchantTrx->details       = 'Payment successful from';
        $merchantTrx->receiver_id   = $user->id;
        $merchantTrx->receiver_type = "USER";
        $merchantTrx->trx           = $senderTrx->trx;
        $merchantTrx->save();

        referralCommission($user, $details->amount,  $senderTrx->trx, 'payment_charge');

        notify($merchant, 'MERCHANT_PAYMENT', [
            'amount'  => showAmount($details->merchant_total_amount, currencyFormat: false),
            'charge'  => showAmount($details->merchant_total_charge, currencyFormat: false),
            'user'    => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'     => $senderTrx->trx,
            'time'    => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($merchant->balance, currencyFormat: false),
        ]);

        notify($user, 'MAKE_PAYMENT', [
            'amount'   => showAmount($details->user_total_amount, currencyFormat: false),
            'charge'   => showAmount($details->user_total_charge, currencyFormat: false),
            'user'     => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'      => $senderTrx->trx,
            'time'     => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance'  => showAmount($user->balance, currencyFormat: false),
            'merchant' => $merchant->fullname . ' ( ' . $merchant->username . ' )',
        ]);
        $notify[] = 'Successfully complete the make payment process';
        return responseSuccess('make_payment_done', $notify, [
            'make_payment' => $senderTrx->load('receiverMerchant'),
        ]);
    }

    public function makePaymentHistory()
    {
        $notify[] = "Make Payment History";
        return responseSuccess('make_payment_history', $notify, [
            'history' => Transaction::where('user_id', auth()->id())->where('user_type', 'USER')->where('remark', 'make_payment')->with('receiverMerchant')->apiQuery(),
        ]);
    }
}
