<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\Invoice;
use App\Models\Merchant;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices      = Invoice::with(['user:id,username,firstname,lastname,email,dial_code,mobile', 'merchant:id,username,email,mobile', 'invoiceItem'])->where('user_id', auth()->id())->latest()->paginate(getPaginate());
        $user          = auth()->user();
        $paymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        $notify[]      = 'Invoice All';
        return responseSuccess('invoice_all', $notify, [
            'otp_type'        => otpType(),
            'current_balance' => $user->balance,
            'payment_charge'  => $paymentCharge,
            'invoices'        => $invoices,
        ]);
    }

    public function payment(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'pin'      => 'required',
            'otp_type' => otpType(validation: true),
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();

        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('not_match', $notify);
        }

        $invoice = Invoice::unpaid()->where('user_id', $user->id)->find($id);

        if (!$invoice) {
            $notify[] = 'Invalid Invoice';
            return responseError('not_found', $notify);
        }

        $makePaymentCharge = TransactionCharge::where('slug', 'payment_charge')->first();
        if (!$makePaymentCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        $amount = $invoice->total_amount;

        $merchantFixedCharge   = $makePaymentCharge->merchant_payment_fixed_charge;
        $merchantPercentCharge = $amount * $makePaymentCharge->merchant_payment_percent_charge / 100;
        $merchantTotalCharge   = $merchantFixedCharge + $merchantPercentCharge;
        $cap                   = $makePaymentCharge->cap;

        if ($makePaymentCharge->cap != -1 && $merchantTotalCharge > $cap) {
            $merchantTotalCharge = $cap;
        }

        $userFixedCharge   = $makePaymentCharge->user_payment_fixed_charge;
        $userPercentCharge = $amount * $makePaymentCharge->user_payment_percent_charge / 100;
        $userTotalCharge   = $userFixedCharge + $userPercentCharge;

        if ($makePaymentCharge->cap != -1 && $userTotalCharge > $cap) {
            $userTotalCharge = $cap;
        }

        $merchantTotalCharge = getAmount($merchantTotalCharge);
        $merchantTotalAmount = getAmount($amount - $merchantTotalCharge);

        $userTotalCharge = getAmount($userTotalCharge);
        $userTotalAmount = getAmount($amount + $userTotalCharge);

        if ($userTotalAmount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('invalid_balance', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'invoice_payment';

        $userAction->details = [
            'amount'                => $amount,
            'merchant_total_amount' => $merchantTotalAmount,
            'merchant_total_charge' => $merchantTotalCharge,
            'user_total_amount'     => $userTotalAmount,
            'user_total_charge'     => $userTotalCharge,
            'invoice_id'            => $invoice->id,
            'done_route'            => 'api.invoice.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'verify_otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function invoiceDone($actionId)
    {
        $user       = auth()->user();
        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;
        $invoice = Invoice::unpaid()->where('user_id', auth()->id())->find($details->invoice_id);

        if (!$invoice) {
            $notify[] = 'Invalid Invoice';
            return responseError('not_found', $notify);
        }

        $merchant = Merchant::where('id', $invoice->merchant_id)->first();
        if (!$merchant) {
            $notify[] = 'Sorry! Merchant not found';
            return responseError('validation_error', $notify);
        }

        if (@$details->user_total_amount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $invoice->status = Status::PAID;
        $invoice->save();

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->user_total_amount;
        $user->save();

        $senderTrx                = new Transaction();
        $senderTrx->user_id       = $user->id;
        $senderTrx->user_type     = 'USER';
        $senderTrx->before_charge = $details->amount;
        $senderTrx->amount        = $details->user_total_amount;
        $senderTrx->post_balance  = $user->balance;
        $senderTrx->charge        = $details->user_total_charge;
        $senderTrx->charge_type   = '+';
        $senderTrx->trx_type      = '-';
        $senderTrx->remark        = 'invoice_payment';
        $senderTrx->details       = 'payment for invoice #' . $invoice->uid;
        $senderTrx->receiver_id   = $merchant->id;
        $senderTrx->receiver_type = "MERCHANT";
        $senderTrx->trx           = $invoice->uid;
        $senderTrx->save();

        generatePoints($senderTrx, $user);

        checkUserReward($user, $senderTrx);

        $merchant->balance += $details->merchant_total_amount;
        $merchant->save();

        $merchantTrx                = new Transaction();
        $merchantTrx->user_id       = $merchant->id;
        $merchantTrx->user_type     = 'MERCHANT';
        $merchantTrx->before_charge = $details->amount;
        $merchantTrx->amount        = $details->merchant_total_amount;
        $merchantTrx->post_balance  = $merchant->balance;
        $merchantTrx->charge        = $details->merchant_total_charge;
        $merchantTrx->charge_type   = '-';
        $merchantTrx->trx_type      = '+';
        $merchantTrx->remark        = 'receive_payment';
        $merchantTrx->details       = 'Payment received for invoice #' . $invoice->uid;
        $merchantTrx->receiver_id   = $user->id;
        $merchantTrx->receiver_type = "USER";
        $merchantTrx->trx           = $invoice->uid;
        $merchantTrx->save();

        referralCommission($user, $details->amount, $invoice->uid, 'payment_charge');

        notify($merchant, 'MERCHANT_PAYMENT', [
            'amount'  => showAmount($details->merchant_total_amount, currencyFormat: false),
            'charge'  => showAmount($details->merchant_total_charge, currencyFormat: false),
            'user'    => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'     => $invoice->uid,
            'time'    => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($merchant->balance, currencyFormat: false),
        ]);

        notify($user, 'MAKE_PAYMENT', [
            'amount'   => showAmount($details->user_total_amount, currencyFormat: false),
            'charge'   => showAmount($details->user_total_charge, currencyFormat: false),
            'user'     => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'      => $senderTrx->trx,
            'time'     => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance'  => showAmount($user->balance, currencyFormat: false),
            'merchant' => $merchant->fullname . ' ( ' . $merchant->username . ' )',
        ]);

        $notify[] = 'Successfully complete the make payment process';
        return responseSuccess('make_payment_done', $notify, [
            'make_payment' => $senderTrx->load('receiverMerchant'),
        ]);
    }

    public function reject(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'reject_reason' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $invoice = Invoice::unpaid()->where('user_id', auth()->id())->find($id);
        if (!$invoice) {
            $notify[] = 'Invalid invoice';
            return responseError('invalid_invoice', $notify);
        }

        $invoice->cancel_reason = $request->reject_reason;
        $invoice->status        = Status::REJECTED;
        $invoice->save();

        $notify[] = 'Invoice rejected successfully';
        return responseSuccess('rejected_invoice', $notify, [
            'invoice' => $invoice,
        ]);
    }

    public function download($id)
    {
        $pageTitle = 'Download Invoice';
        $invoice   = Invoice::with(['user', 'merchant'])->where('user_id', auth()->id())->find($id);

        if (!$invoice) {
            $notify[] = 'Invoice not found';
            return responseError('not_found', $notify);
        }

        $pdf = Pdf::loadView('invoice.download', compact('invoice', 'pageTitle'));

        return response()->streamDownload(
            fn() => print($pdf->output()),
            'invoice_' . $invoice->uid . '.pdf',
            [
                'Content-Type'        => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="invoice_' . $invoice->uid . '.pdf"',
            ]
        );
    }
}
