<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\SetupDonation;
use App\Models\Transaction;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class DonationController extends Controller
{
    public function donation()
    {
        $notify[] = "Donation";

        $user           = auth()->user();
        $setupDonations = SetupDonation::active()->get();

        return responseSuccess('donation', $notify, [
            'otp_type'        => otpType(),
            'current_balance' => $user->balance,
            'setup_donations' => $setupDonations,
        ]);
    }

    public function donationConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'            => 'required|gt:0',
            'setup_donation_id' => 'required|gt:0',
            'name'              => 'required',
            'email'             => 'required|email',
            'otp_type'          => otpType(validation: true),
            'pin'               => 'required',
            'hide_identity'     => 'required|in:0,1',
            'reference'         => 'nullable',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $setup = SetupDonation::active()->find($request->setup_donation_id);
        if (!$setup) {
            $notify[] = 'Sorry, Setup donation not found';
            return responseError('validation_error', $notify);
        }

        if ($request->amount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'donation';

        $userAction->details = [
            'amount'            => $request->amount,
            'hide_identity'     => $request->hide_identity,
            'setup_donation_id' => $setup->id,
            'name'              => $request->name,
            'email'             => $request->email,
            'reference'         => $request->reference,
            'done_route'        => 'api.donation.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();

        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('validation_error', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function donationDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;
        if (@$details->amount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $setup = SetupDonation::find($details->setup_donation_id);

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->amount;
        $user->save();

        $transaction                    = new Transaction();
        $transaction->setup_donation_id = $setup->id;
        $transaction->user_id           = $user->id;
        $transaction->user_type         = 'USER';
        $transaction->before_charge     = $details->amount;
        $transaction->amount            = $details->amount;
        $transaction->post_balance      = $user->balance;
        $transaction->charge            = 0;
        $transaction->charge_type       = '+';
        $transaction->trx_type          = '-';
        $transaction->remark            = 'donation';
        $transaction->details           = 'Donation';
        $transaction->receiver_id       = 0;
        $transaction->receiver_type     = null;
        $transaction->trx               = getTrx();
        $transaction->reference         = $details->reference;
        $transaction->hide_identity     = $details->hide_identity;
        $transaction->save();

        generatePoints($transaction, $user);

        checkUserReward($user, $transaction);

        notify($user, 'DONATION', [
            'amount'       => showAmount($details->amount, currencyFormat: false),
            'name'         => $details->name,
            'email'        => $details->email,
            'donation_for' => $setup->name,
            'trx'          => $transaction->trx,
            'time'         => showDateTime($transaction->created_at, 'd/M/Y @h:i a'),
            'balance'      => showAmount($user->balance, currencyFormat: false),
        ]);

        $notify[] = 'Successfully complete the donation process';
        return responseSuccess('donation_done', $notify, [
            'donation' => $transaction->load('donationFor'),
        ]);
    }

    public function donationHistory()
    {
        $notify[] = "Donation History";
        return responseSuccess('donation_history', $notify, [
            'history' => Transaction::where('user_id', auth()->id())->where('remark', 'donation')->with('donationFor')->apiQuery(),
        ]);
    }
}
