<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\Agent;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CashOutController extends Controller
{

    public function checkUser(Request $request)
    {
        $agent = Agent::where('username', $request->agent)->orWhere('mobile', $request->agent)->first();
        if (!$agent) {
            $notify[] = 'Agent not found';
            return responseError('validation_error', $notify);
        }

        $notify[] = 'Check Agent';
        return responseSuccess('check_agent', $notify, [
            'agent' => $agent,
        ]);
    }

    public function cashOut()
    {
        $notify[] = "Cash Out";

        $user          = auth()->user();
        $cashOutCharge = TransactionCharge::where('slug', 'cash_out')->first();

        return responseSuccess('cash_out', $notify, [
            'otp_type'                => otpType(),
            'current_balance'         => $user->balance,
            'cash_out'         => $cashOutCharge,
            'latest_cash_out_history' => Transaction::where('user_id', $user->id)
                ->where('user_type', 'USER')
                ->where('remark', 'cash_out')
                ->with('receiverAgent')
                ->groupBy('receiver_id')
                ->orderBy('id', 'DESC')
                ->limit(10)
                ->get(),
        ]);
    }

    public function cashOutConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'   => 'required|gt:0',
            'agent'    => 'required',
            'otp_type' => otpType(validation: true),
            'pin'      => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        //Find agent
        $agent = Agent::where('username', $request->agent)->orWhere('mobile', $request->agent)->first();
        if (!$agent) {
            $notify[] = 'Agent not found';
            return responseError('validation_error', $notify);
        }

        $cashOutCharge = TransactionCharge::where('slug', 'cash_out')->first();
        if (!$cashOutCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        if ($request->amount < $cashOutCharge->min_limit || $request->amount > $cashOutCharge->max_limit) {
            $notify[] = 'Please Follow the cash out limit';
            return responseError('validation_error', $notify);
        }

        $fixedCharge   = $cashOutCharge->fixed_charge;
        $percentCharge = $request->amount * $cashOutCharge->percent_charge / 100;
        $totalCharge   = $fixedCharge + $percentCharge;

        //Agent commission
        $fixedCommission   = $cashOutCharge->agent_commission_fixed;
        $percentCommission = $request->amount * $cashOutCharge->agent_commission_percent / 100;

        $totalAmount = getAmount($request->amount + $totalCharge);

        if ($cashOutCharge->daily_limit != -1 && ($user->trxLimit('cash_out')['daily'] + $totalAmount) > $cashOutCharge->daily_limit) {
            $notify[] = 'Your daily cash out limit exceeded';
            return responseError('validation_error', $notify);
        }

        if ($cashOutCharge->monthly_limit != 1 && ($user->trxLimit('cash_out')['monthly'] + $totalAmount) > $cashOutCharge->monthly_limit) {
            $notify[] = 'Your monthly cash out limit exceeded';
            return responseError('validation_error', $notify);
        }

        $totalCommission = getAmount($fixedCommission + $percentCommission);
        $totalCharge     = getAmount($totalCharge);

        if ($totalAmount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'cash_out';

        $userAction->details = [
            'amount'           => $request->amount,
            'total_amount'     => $totalAmount,
            'total_charge'     => $totalCharge,
            'agent_id'         => $agent->id,
            'total_commission' => $totalCommission,
            'done_route'       => 'api.cash.out.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function cashOutDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->where('id', $actionId)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;

        $agent = Agent::where('id', $details->agent_id)->first();
        if (!$agent) {
            $notify[] = 'Sorry! Agent not found';
            return responseError('validation_error', $notify);
        }

        if (@$userAction->details->totalAmount > $user->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction->is_used = Status::YES;
        $userAction->save();

        $user->balance -= $details->total_amount;
        $user->save();

        $senderTrx                = new Transaction();
        $senderTrx->user_id       = $user->id;
        $senderTrx->user_type     = 'USER';
        $senderTrx->before_charge = $details->amount;
        $senderTrx->amount        = $details->total_amount;
        $senderTrx->post_balance  = $user->balance;
        $senderTrx->charge        = $details->total_charge;
        $senderTrx->charge_type   = '+';
        $senderTrx->trx_type      = '-';
        $senderTrx->remark        = 'cash_out';
        $senderTrx->details       = 'Cash out to';
        $senderTrx->receiver_id   = $agent->id;
        $senderTrx->receiver_type = "AGENT";
        $senderTrx->trx           = getTrx();
        $senderTrx->save();

        generatePoints($senderTrx, $user);

        checkUserReward($user, $senderTrx);

        $agent->balance += $details->amount;
        $agent->save();

        $agentTrx                = new Transaction();
        $agentTrx->user_id       = $agent->id;
        $agentTrx->user_type     = 'AGENT';
        $agentTrx->before_charge = $details->amount;
        $agentTrx->amount        = $details->amount;
        $agentTrx->post_balance  = $agent->balance;
        $agentTrx->charge        = 0;
        $agentTrx->charge_type   = '+';
        $agentTrx->trx_type      = '+';
        $agentTrx->remark        = 'cash_out';
        $agentTrx->details       = 'Cash out from ';
        $agentTrx->receiver_id   = $user->id;
        $agentTrx->receiver_type = "USER";
        $agentTrx->trx           = $senderTrx->trx;
        $agentTrx->save();

        if ($details->total_commission) {
            //Agent commission
            $agent->balance += $details->total_commission;
            $agent->save();

            $agentCommissionTrx                = new Transaction();
            $agentCommissionTrx->user_id       = $agent->id;
            $agentCommissionTrx->user_type     = 'AGENT';
            $agentCommissionTrx->before_charge = $details->total_commission;
            $agentCommissionTrx->amount        = $details->total_commission;
            $agentCommissionTrx->post_balance  = $agent->balance;
            $agentCommissionTrx->charge        = 0;
            $agentCommissionTrx->charge_type   = '+';
            $agentCommissionTrx->trx_type      = '+';
            $agentCommissionTrx->remark        = 'commission';
            $agentCommissionTrx->details       = 'Cash out commission';
            $agentCommissionTrx->trx           = $senderTrx->trx;
            $agentCommissionTrx->save();

            //Agent commission
            notify($agent, 'CASH_OUT_COMMISSION_AGENT', [
                'amount'     => showAmount($details->amount, currencyFormat: false),
                'commission' => showAmount($details->total_commission, currencyFormat: false),
                'trx'        => $senderTrx->trx,
                'time'       => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
                'balance'    => showAmount($agent->balance, currencyFormat: false),
            ]);
        }

        referralCommission($user, $details->amount, $senderTrx->trx, 'cash_out');

        //To user
        notify($user, 'CASH_OUT', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'charge'  => showAmount($details->total_charge, currencyFormat: false),
            'agent'   => $agent->fullname . ' ( ' . $agent->username . ' )',
            'trx'     => $senderTrx->trx,
            'time'    => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($user->balance, currencyFormat: false),
        ]);

        //To agent
        notify($agent, 'CASH_OUT_TO_AGENT', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'user'    => $user->fullname . ' ( ' . $user->username . ' )',
            'trx'     => $senderTrx->trx,
            'time'    => showDateTime($senderTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($agent->balance, currencyFormat: false),
        ]);
        $notify[] = 'Successfully complete the cash out process';
        return responseSuccess('cash_out_done', $notify, [
            'cash_out' => $senderTrx->load('receiverAgent'),
        ]);
    }

    public function cashOutHistory()
    {
        $notify[] = "Cash Out History";
        return responseError('cash_out_history', $notify, [
            'history' => Transaction::where('user_id', auth()->id())->where('user_type', 'USER')->where('remark', 'cash_out')->with('receiverAgent')->apiQuery(),
        ]);
    }
}
