<?php

namespace App\Http\Controllers\Api\Auth;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use App\Models\User;
use App\Models\UserLogin;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
     */

    use RegistersUsers;

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        $agree = 'nullable';
        if (gs('agree')) {
            $agree = 'required';
        }

        $countryData  = (array) json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $validate = Validator::make($data, [
            'firstname'    => 'required',
            'lastname'     => 'required',
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', @$data['mobile_code'])],
            'email'        => 'required|string|email|unique:users',
            'password'     => ['required', 'confirmed', 'digits:4'],
            'agree'        => $agree,
        ], [
            'firstname.required' => 'The first name field is required',
            'lastname.required'  => 'The last name field is required',
        ]);

        return $validate;
    }

    public function register(Request $request)
    {
        if (!gs('registration')) {
            $notify[] = 'Registration not allowed';
            return responseError('validation_error', $notify);
        }

        $validator = $this->validator($request->all());
        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = $this->create($request->all());

        $response['access_token'] = $user->createToken('auth_token')->plainTextToken;
        $response['user']         = $user;
        $response['token_type']   = 'Bearer';
        $notify[]                 = 'Registration successful';

        return responseSuccess('registration_success', $notify, $response);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array $data
     * @return \App\User
     */
    protected function create(array $data)
    {

        $referBy = $data['reference'] ?? null;
        if ($referBy && gs('referral')) {
            $referUser = User::where('username', $referBy)->first();
        } else {
            $referUser = null;
        }

        //User Create
        $user               = new User();
        $user->ref_by       = $referUser ? $referUser->id : 0;
        $user->firstname    = $data['firstname'];
        $user->lastname     = $data['lastname'];
        $user->dial_code    = $data['mobile_code'];
        $user->country_code = $data['country_code'];
        $user->mobile       = $data['mobile'];
        $user->country_name = $data['country'];
        $user->email        = strtolower($data['email']);
        $user->password     = Hash::make($data['password']);
        $user->kv           = gs('kv') ? Status::UNVERIFIED : Status::VERIFIED;
        $user->ev           = gs('ev') ? Status::UNVERIFIED : Status::VERIFIED;
        $user->sv           = gs('sv') ? Status::UNVERIFIED : Status::VERIFIED;
        $user->ts           = Status::DISABLE;
        $user->tv           = Status::VERIFIED;
        $user->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = 'New member registered';
        $adminNotification->click_url = urlPath('admin.users.detail', $user->id);
        $adminNotification->save();

        //Login Log Create
        $ip        = getRealIP();
        $exist     = UserLogin::where('user_ip', $ip)->first();
        $userLogin = new UserLogin();

        //Check exist or not
        if ($exist) {
            $userLogin->longitude    = $exist->longitude;
            $userLogin->latitude     = $exist->latitude;
            $userLogin->city         = $exist->city;
            $userLogin->country_code = $exist->country_code;
            $userLogin->country      = $exist->country;
        } else {
            $info                    = json_decode(json_encode(getIpInfo()), true);
            $userLogin->longitude    = @implode(',', $info['long']);
            $userLogin->latitude     = @implode(',', $info['lat']);
            $userLogin->city         = @implode(',', $info['city']);
            $userLogin->country_code = @implode(',', $info['code']);
            $userLogin->country      = @implode(',', $info['country']);
        }

        $userAgent          = osBrowser();
        $userLogin->user_id = $user->id;
        $userLogin->user_ip = $ip;

        $userLogin->browser = @$userAgent['browser'];
        $userLogin->os      = @$userAgent['os_platform'];
        $userLogin->save();

        $user = User::find($user->id);

        return $user;
    }
}
