<?php

namespace App\Http\Controllers\Api;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Lib\Reloadly;
use App\Models\Country;
use App\Models\Operator;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AirtimeController extends Controller
{
    public function countries()
    {
        $user = auth()->user();
        $countries = Country::active()->whereHas('operators', function ($query) {
            $query->active();
        })->get();

        $airtimeCharge = TransactionCharge::where('slug', 'airtime_charge')->first();

        $notify[] = 'Airtime Top-Up Countries';
        return responseSuccess('countries', $notify, [
            'countries' => $countries,
            'user'      => $user,
            'charge'    => $airtimeCharge,
        ]);

        return view('Template::user.airtime.form', compact('pageTitle', 'countries'));
    }

    public function getOperatorByCountry($id)
    {
        $operators = Operator::active()->where('country_id', $id)->get();
        $notify[]  = 'Get operator by country';

        return responseSuccess('operator_by_country', $notify, [
            'operators' => $operators,
            'otp_type'  => otpType(),
        ]);
    }

    public function topUpConfirm(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'country_id'    => 'required|integer',
            'operator_id'   => 'required|integer',
            'calling_code'  => 'required|string',
            'mobile_number' => 'required|numeric',
            'amount'        => 'required|numeric|gt:0',
            'otp_type'      => otpType(validation: true),
            'pin'           => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user();
        if (!Hash::check($request->pin, $user->password)) {
            $notify[] = 'Provided PIN does not correct';
            return responseError('validation_error', $notify);
        }

        $operator = Operator::active()->find($request->operator_id);

        if (!$operator) {
            $notify[] = 'Invalid operator selected';
            return responseError('invalid_operator', $notify);
        }

        if ($operator->denomination_type == 'FIXED') {
            if (!in_array($request->amount, $operator->fixed_amounts)) {
                $notify[] = 'Invalid amount selected';
                return responseError('invalid_operator', $notify);
            }
        } else {
            $minAmount = $operator->min_amount;
            $maxAmount = $operator->max_amount;

            if ($request->amount < $minAmount) {
                $notify[] = 'Amount should be greater than ' . $minAmount . ' ' . gs('cur_text');
                return responseError('invalid_operator', $notify);
            }

            if ($request->amount > $maxAmount) {
                $notify[] = 'Amount should be less than ' . $maxAmount . ' ' . gs('cur_text');
                return responseError('invalid_operator', $notify);
            }
        }

        $callingCode  = $request->calling_code;
        $mobileNumber = $request->mobile_number;

        $country = Country::active()->find($request->country_id);
        if (!$country) {
            $notify[] = 'Country not found';
            return responseError('country_not_found', $notify);
        }

        $airtimeCharge = TransactionCharge::where('slug', 'airtime_charge')->first();
        if (!$airtimeCharge) {
            $notify[] = 'Sorry, Transaction charge not found';
            return responseError('validation_error', $notify);
        }

        $fixedCharge = $airtimeCharge->fixed_charge;
        $totalCharge = ($request->amount * $airtimeCharge->percent_charge / 100) + $fixedCharge;
        $cap         = $airtimeCharge->cap;

        if ($airtimeCharge->cap != -1 && $totalCharge > $cap) {
            $totalCharge = $cap;
        }

        $totalAmount = getAmount($request->amount + $totalCharge);

        if ($totalAmount > $user->balance) {
            $notify[] = 'Insufficient balance';
            return responseError('insufficient_balance', $notify);
        }

        $userAction          = new UserActionProcess();
        $userAction->user_id = $user->id;
        $userAction->act     = 'air_time';

        $userAction->details = [
            'amount'             => $request->amount,
            'total_amount'       => $totalAmount,
            'total_charge'       => $totalCharge,
            'mobile_number'      => $mobileNumber,
            'country_iso_name'   => $country->iso_name,
            'calling_code'       => $callingCode,
            'operator_unique_id' => $operator->unique_id,
            'operator_id'        => $operator->id,
            'done_route'         => 'api.top.up.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }
        $userAction->submit();

        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function topUpDone($actionId)
    {
        $user = auth()->user();

        $userAction = UserAction::where('id', $actionId)
            ->where('user_id', $user->id)
            ->where('user_type', 'USER')
            ->where('is_api', Status::YES)
            ->where('is_used', Status::NO)
            ->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;

        $recipient['number']      = $details->mobile_number;
        $recipient['countryCode'] = $details->country_iso_name;

        $reloadly             = new Reloadly();
        $reloadly->operatorId = $details->operator_unique_id;

        $response = $reloadly->topUp($details->amount, $recipient);

        if ($response['status']) {
            $user->balance -= $details->total_amount;
            $user->save();

            $transaction                = new Transaction();
            $transaction->operator_id   = $details->operator_id;
            $transaction->user_id       = $user->id;
            $transaction->user_type     = 'USER';
            $transaction->before_charge = $details->amount;
            $transaction->amount        = $details->total_amount;
            $transaction->charge        = $details->total_charge;
            $transaction->post_balance  = $user->balance;
            $transaction->trx_type      = '-';
            $transaction->trx           = $response['custom_identifier'] ?? getTrx();
            $transaction->details       = 'Top-up ' . $details->amount . ' ' . gs('cur_text') . ' to ' . $details->calling_code . $details->mobile_number;
            $transaction->remark        = 'top_up';
            $transaction->save();

            generatePoints($transaction, $user);

            checkUserReward($user, $transaction);

            notify($user, 'TOP_UP', [
                'amount'        => showAmount($details->amount, currencyFormat: false),
                'mobile_number' => $details->calling_code . $details->mobile_number,
                'post_balance'  => showAmount($user->balance, currencyFormat: false),
            ]);

            $notify[] = 'Top-Up completed successfully';
            return responseSuccess('top_up_success', $notify);
        } else {
            $notify[] = @$response['message'];
            return responseError('something_wrong', $notify);
        }
    }

    public function history()
    {
        $transactions = Transaction::with('operator')->where('user_id', auth()->id())->where('remark', 'top_up')->apiQuery();
        $notify[]     = 'Airtime history';
        return responseSuccess('airtime_history', $notify, [
            'transactions' => $transactions,
        ]);
    }
}
