<?php

namespace App\Http\Controllers\Api\Agent;

use App\Constants\Status;
use App\Traits\Common;
use App\Http\Controllers\Controller;
use App\Lib\Api\UserActionProcess;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\User;
use App\Models\UserAction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CashInController extends Controller
{

    use Common;

    public function checkUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = User::where('username', $request->user)->orWhereRaw("CONCAT(dial_code, mobile) = ?", [$request->user])->first();
        if (!$user) {
            $notify[] = 'User not found';
            return responseError('validation_error', $notify);
        }
        $notify[] = 'Check user';
        return responseSuccess('check_user', $notify, [
            'user' => $user,
        ]);
    }

    public function cashInForm()
    {
        $notify[]     = "Cash In";
        $cashInCharge = TransactionCharge::where('slug', 'cash_in')->first();
        return responseSuccess('cash_in', $notify, [
            'otp_type'       => otpType(),
            'cash_in' => $cashInCharge,
        ]);
    }

    public function withCashInLimit($wallets, $moneyOutCharge)
    {
        foreach ($wallets ?? [] as $wallet) {

            $rate = $wallet->currency->rate;

            $min = $moneyOutCharge->min_limit / $rate;
            $max = $moneyOutCharge->max_limit / $rate;

            $wallet->currency->money_out_min_limit = $min;
            $wallet->currency->money_out_max_limit = $max;
        }

        return $wallets;
    }

    public function confirmCashIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount'   => 'required|gt:0',
            'user'     => 'required',
            'otp_type' => otpType(validation: true),
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $cashInCharge = TransactionCharge::where('slug', 'cash_in')->first();

        if (!$cashInCharge) {
            $notify[] = 'Transaction charge not found';
            return responseError('validation_error', $notify);
            return response()->json([
                'remark'  => 'validation_error',
                'status'  => 'error',
                'message' => ['error' => ['Sorry, Transaction charge not found']],
            ]);
        }

        $agent = auth()->user('agent');

        if ($cashInCharge->daily_limit != -1 && $agent->trxLimit('cash_in')['daily'] > $cashInCharge->daily_limit) {
            $notify[] = 'Your daily money in limit exceeded';
            return responseError('validation_error', $notify);
        }

        if ($cashInCharge->monthly_limit != 1 && $agent->trxLimit('money_out')['monthly'] > $cashInCharge->monthly_limit) {
            $notify[] = 'Your monthly money in limit exceeded';
            return responseError('validation_error', $notify);
        }

        $user = User::where('username', $request->user)->orWhereRaw("CONCAT(dial_code, mobile) = ?", [$request->user])->first();
        if (!$user) {
            $notify[] = 'Sorry! User not found';
            return responseError('validation_error', $notify);
        }

        if ($request->amount < $cashInCharge->min_limit || $request->amount > $cashInCharge->max_limit) {
            $notify[] = 'Please Follow the money in limit';
            return responseError('validation_error', $notify);
        }

        //Agent commission
        $fixedCommission   = $cashInCharge->agent_commission_fixed;
        $percentCommission = $request->amount * $cashInCharge->agent_commission_percent / 100;
        $totalCommission   = $fixedCommission + $percentCommission;

        if ($request->amount > $agent->balance) {
            $notify[] = 'Sorry! Insufficient balance in your wallet';
            return responseError('validation_error', $notify);
        }

        $recentSimilarCashin = Transaction::where('user_id', $agent->id)
            ->where('remark', 'cash_in')
            ->where('receiver_id', $user->id)
            ->where('amount', $request->amount)
            ->orderBy('id', 'desc')
            ->where('created_at', '>', now()->subMinutes(3))
            ->first();
        if ($recentSimilarCashin) {
            $notify[] = ['Request cannot be made.Please wait for 3 minutes at least'];
            return responseError('validation_error', $notify);
        }

        $userAction            = new UserActionProcess();
        $userAction->user_id   = $this->guard()['user']->id;
        $userAction->user_type = $this->guard()['user_type'];
        $userAction->act       = 'cash_in';

        $userAction->details = [
            'user_id'         => $user->id,
            'amount'          => $request->amount,
            'totalCommission' => $totalCommission,
            'done_route'      => 'api.agent.cash.in.done',
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }

        $userAction->submit();
        $actionId = $userAction->action_id;

        if ($userAction->verify_api_otp) {
            $notify[] = 'Verify otp';
            return responseSuccess('verify_otp', $notify, [
                'action_id' => $actionId,
            ]);
        }

        return callApiMethod($userAction->next_route, $actionId);
    }

    public function cashInDone($actionId)
    {
        $userAction = UserAction::where('user_id', $this->guard()['user']->id)->where('user_type', 'AGENT')->where('is_api', 1)->where('is_used', 0)->where('id', $actionId)->first();

        if (!$userAction) {
            $notify[] = 'Sorry! Unable to process';
            return responseError('validation_error', $notify);
        }

        $details = $userAction->details;

        $user = User::where('id', $details->user_id)->first();
        if (!$user) {
            $notify[] = 'Sorry! User not found';
            return responseError('validation_error', $notify);
        }

        $agent = auth()->user('agent');

        if (@$userAction->details->amount > $agent->balance) {
            $notify[] = 'Sorry! Insufficient balance';
            return responseError('validation_error', $notify);
        }

        $userAction->is_used = Status::YES;
        $userAction->save();

        $agent->balance -= $details->amount;
        $agent->save();

        $agentTrx                = new Transaction();
        $agentTrx->user_id       = $agent->id;
        $agentTrx->user_type     = 'AGENT';
        $agentTrx->before_charge = $details->amount;
        $agentTrx->amount        = $details->amount;
        $agentTrx->post_balance  = $agent->balance;
        $agentTrx->charge        = 0;
        $agentTrx->charge_type   = '+';
        $agentTrx->trx_type      = '-';
        $agentTrx->remark        = 'cash_in';
        $agentTrx->details       = 'Cash in to';
        $agentTrx->receiver_id   = $user->id;
        $agentTrx->receiver_type = 'USER';
        $agentTrx->trx           = getTrx();
        $agentTrx->save();

        $user->balance += $details->amount;
        $user->save();

        $userTrx                = new Transaction();
        $userTrx->user_id       = $user->id;
        $userTrx->user_type     = 'USER';
        $userTrx->before_charge = $details->amount;
        $userTrx->amount        = $details->amount;
        $userTrx->post_balance  = $user->balance;
        $userTrx->charge        = 0;
        $userTrx->charge_type   = '+';
        $userTrx->trx_type      = '+';
        $userTrx->remark        = 'cash_in';
        $userTrx->details       = 'Cash in from';
        $userTrx->receiver_id   = $agent->id;
        $userTrx->receiver_type = 'AGENT';
        $userTrx->trx           = $agentTrx->trx;
        $userTrx->save();

        generatePoints($userTrx, $user);

        checkUserReward($user, $userTrx);

        if ($details->totalCommission) {
            $agent->balance += $details->totalCommission;
            $agent->save();

            $agentCommissionTrx                = new Transaction();
            $agentCommissionTrx->user_id       = $agent->id;
            $agentCommissionTrx->user_type     = 'AGENT';
            $agentCommissionTrx->before_charge = $details->totalCommission;
            $agentCommissionTrx->amount        = $details->totalCommission;
            $agentCommissionTrx->post_balance  = $agent->balance;
            $agentCommissionTrx->charge        = 0;
            $agentCommissionTrx->charge_type   = '+';
            $agentCommissionTrx->trx_type      = '+';
            $agentCommissionTrx->remark        = 'commission';
            $agentCommissionTrx->details       = 'Cash in commission';
            $agentCommissionTrx->trx           = $agentTrx->trx;
            $agentCommissionTrx->save();

            //Agent commission
            notify($agent, 'CASH_IN_COMMISSION_AGENT', [
                'amount'     => showAmount($details->amount, currencyFormat: false),
                'commission' => showAmount($details->totalCommission, currencyFormat: false),
                'trx'        => $agentTrx->trx,
                'time'       => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
                'balance'    => showAmount($agent->balance, currencyFormat: false),
            ]);
        }

        //To user
        notify($user, 'CASH_IN', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'agent'   => $agent->username,
            'trx'     => $agentTrx->trx,
            'time'    => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($user->balance, currencyFormat: false),
        ]);

        //To agent
        notify($agent, 'CASH_IN_AGENT', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'user'    => $user->fullname,
            'trx'     => $agentTrx->trx,
            'time'    => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($agent->balance, currencyFormat: false),
        ]);
        $notify[] = 'Cash in successfully';
        return responseSuccess('cash_in_done', $notify);
    }
}
