<?php

namespace App\Http\Controllers\Api\Agent\Auth;

use App\Models\UserLogin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Support\Facades\Validator;
use App\Constants\Status;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */

    protected $username;

    /**
     * Create a new controller instance.
     *
     * @return void
     */


    public function __construct()
    {
        $this->username = $this->findUsername();
        parent::__construct();
    }


    public function login(Request $request)
    {

        $validator = $this->validateLogin($request);

        if ($request->is_web) {
            if (!verifyCaptcha()) {
                $notify[] = 'Invalid captcha provided';
                return responseError('captcha_error', $notify);
            }
        }

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $credentials = request([$this->username, 'password']);

        if (!Auth::guard('agent')->attempt($credentials)) {
            $notify[] = 'Unauthorized Agent';
            return responseError('validation_error', $notify);
        }

        $user = auth()->guard('agent')->user();

        $tokenResult = $user->createToken('auth_token', ['agent'])->plainTextToken;
        $this->authenticated($request, $user);

        if ($user->is_deleted) {
            $user->tokens()->delete();
            $notify[] = 'Unauthorized Agent';
            return responseError('validation_error', $notify);
        }

        $notify[] = 'Login Successful';
        return responseSuccess('login_success', $notify, [
            'user' => auth()->guard('agent')->user(),
            'access_token' => $tokenResult,
            'token_type' => 'Bearer'
        ]);
    }

    public function findUsername()
    {
        $login = request()->input('username');

        $fieldType = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
        request()->merge([$fieldType => $login]);
        return $fieldType;
    }

    protected function guard()
    {
        return Auth::guard('agent');
    }

    public function username()
    {
        return $this->username;
    }

    protected function validateLogin(Request $request)
    {
        $validation_rule = [
            $this->username() => 'required|string',
            'password' => 'required|string',
        ];

        $validate = Validator::make($request->all(), $validation_rule);
        return $validate;
    }


    public function logout()
    {
        auth()->user('agent')->tokens()->delete();

        $notify[] = 'Logout successfully';
        return responseSuccess('logout_success', $notify);
    }


    public function authenticated(Request $request, $user)
    {
        $user->tv = $user->ts == Status::ENABLE ? Status::DISABLE : Status::ENABLE;
        $user->save();

        $ip = $_SERVER["REMOTE_ADDR"];
        $exist = UserLogin::where('user_ip', $ip)->first();
        $userLogin = new UserLogin();

        if ($exist) {
            $userLogin->longitude =  $exist->longitude;
            $userLogin->latitude =  $exist->latitude;
            $userLogin->city =  $exist->city;
            $userLogin->country_code = $exist->country_code;
            $userLogin->country =  $exist->country;
        } else {
            $info = json_decode(json_encode(getIpInfo()), true);
            $userLogin->longitude =  @implode(',', $info['long']);
            $userLogin->latitude =  @implode(',', $info['lat']);
            $userLogin->city =  @implode(',', $info['city']);
            $userLogin->country_code = @implode(',', $info['code']);
            $userLogin->country =  @implode(',', $info['country']);
        }

        $userAgent = osBrowser();
        $userLogin->agent_id = $user->id;
        $userLogin->user_ip =  $ip;

        $userLogin->browser = @$userAgent['browser'];
        $userLogin->os = @$userAgent['os_platform'];
        $userLogin->save();
    }
}
