<?php

namespace App\Http\Controllers\Api\Agent;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Models\Deposit;
use App\Models\DeviceToken;
use App\Models\Form;
use App\Models\QrCode;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\Withdrawal;
use App\Rules\FileTypeValidate;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class AgentController extends Controller
{

    public function home()
    {
        $user = auth()->user('agent');
        $totalAddMoney = Deposit::where('user_id', $user->id)->where('user_type', 'AGENT')->where('status', Status::PAYMENT_SUCCESS)->sum('amount');
        $totalWithdraw = Withdrawal::where('user_id', $user->id)->where('user_type', 'AGENT')->where('status', Status::PAYMENT_SUCCESS)->sum('amount');

        $latestTrx = Transaction::where('user_id', $user->id)->where('user_type', 'AGENT')
            ->with('receiverUser', 'receiverAgent', 'receiverMerchant')->orderBy('id', 'desc')->take(10)
            ->get();

        $date    = Carbon::today()->subDays(7);
        $moneyIn = Transaction::where('user_id', $user->id)->where('user_type', 'AGENT')->where('trx_type', '+')->whereDate('created_at', '>=', $date)->get(['amount']);

        $moneyOut = Transaction::where('user_id', $user->id)->where('user_type', 'AGENT')->where('trx_type', '-')->whereDate('created_at', '>=', $date)->get(['amount']);

        $totalMoneyIn  = 0;
        $totalMoneyOut = 0;

        $in = [];
        foreach ($moneyIn as $inTrx) {
            $in[] = $inTrx->amount;
        }
        $totalMoneyIn = array_sum($in);

        $out = [];
        foreach ($moneyOut as $outTrx) {
            $out[] = $outTrx->amount;
        }

        $totalMoneyOut = array_sum($out);

        $totalMoneyInOut = ['totalMoneyIn' => $totalMoneyIn, 'totalMoneyOut' => $totalMoneyOut];

        $notify[] = 'Dashboard';
        return responseSuccess('dashboard', $notify, [
            'agent'                   => $user,
            'latest_trx'              => $latestTrx,
            'last_7_day_money_in_out' => $totalMoneyInOut,
            'total_add_money'         => showAmount($totalAddMoney ?? 0),
            'total_withdraw'          => showAmount($totalWithdraw ?? 0),
        ]);
    }

    public function userInfo()
    {
        $notify[] = 'User information';
        return responseSuccess('user_info', $notify, [
            'user' => auth()->user('agent'),
        ]);
    }

    public function submitProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'firstname' => 'required',
            'lastname'  => 'required',
            'image'     => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = agent();

        $user->firstname = $request->firstname;
        $user->lastname  = $request->lastname;
        $user->address   = [
            'country' => @$user->address->country,
            'address' => $request->address,
            'state'   => $request->state,
            'zip'     => $request->zip,
            'city'    => $request->city,
        ];

        if ($request->hasFile('image')) {
            try {
                $old         = $user->image;
                $user->image = fileUploader($request->image, getFilePath('agentProfile'), getFileSize('agentProfile'), $old);
            } catch (\Exception $exp) {
                $notify[] = 'Couldn\'t upload your image';
                return responseError('validation_error', $notify);
            }
        }

        $user->save();

        $notify[] = 'Profile updated successfully';
        return responseSuccess('profile_updated', $notify);
    }

    public function submitPassword(Request $request)
    {
        $passwordValidation = Password::min(6);
        $general            = gs();
        if ($general->secure_password) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'password'         => ['required', 'confirmed', $passwordValidation],
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user('agent');
        if (Hash::check($request->current_password, $user->password)) {
            $password       = Hash::make($request->password);
            $user->password = $password;
            $user->save();
            $notify[] = 'Password changed successfully';
            return responseSuccess('password_changed', $notify);
        } else {
            $notify[] = 'The password doesn\'t match!';
            return responseError('validation_error', $notify);
        }
    }

    public function transactions(Request $request)
    {
        $times        = ['7days', '15days', '30days', '365days'];
        $remarks      = Transaction::where('user_type', 'AGENT')->distinct('remark')->orderBy('remark')->pluck('remark')->toArray();
        $transactions = Transaction::where('user_type', 'AGENT')->where('user_id', auth()->id());

        if ($request->search) {
            $transactions = $transactions->where('trx', $request->search);
        }

        if ($request->type) {
            $type         = $request->type == 'plus' ? '+' : '-';
            $transactions = $transactions->where('trx_type', $type);
        }

        if ($request->remark) {
            $transactions = $transactions->where('remark', $request->remark);
        }

        if ($request->days) {
            $transactions = $transactions->where('created_at', '>=', now()->subDays($request->days));
        }

        $transactions = $transactions->orderBy('id', 'desc')->apiQuery();
        $notify[]     = 'Transactions data';
        return responseSuccess('transactions', $notify, [
            'transactions' => $transactions,
            'remarks'      => $remarks,
            'times'        => $times,
        ]);
    }

    public function depositHistory()
    {
        $logs = Deposit::searchable(['trx'])->where('user_id', auth()->user('agent')->id)->where('user_type', 'AGENT')->with('gateway')->orderBy('id', 'desc')->paginate(getPaginate());
        return response()->json([
            'remark' => 'Add money history',
            'stutas' => 'Success',
            'data'   => [
                'logs' => $logs,
            ],
        ]);
    }

    public function kycForm()
    {
        if (auth()->user('agent')->kv == 2) {
            $notify[] = 'Your KYC is under review';
            return responseError('under_review', $notify, [
                'kyc_data' => auth()->user('agent')->kyc_data,
                'path'     => getFilePath('verify'),
            ]);
        }
        if (auth()->user('agent')->kv == 1) {
            $notify[] = 'You are already KYC verified';
            return responseError('already_verified', $notify);
        }
        $form     = Form::where('act', 'agent_kyc')->first();
        $notify[] = 'KYC field is below';
        return responseSuccess('kyc_form', $notify, [
            'form' => $form->form_data,
        ]);
    }

    public function kycSubmit(Request $request)
    {
        $form           = Form::where('act', 'agent_kyc')->first();
        $formData       = $form->form_data;
        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $validator = Validator::make($request->all(), $validationRule);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $userData       = $formProcessor->processFormData($request, $formData);
        $user           = auth()->user('agent');
        $user->kyc_data = $userData;
        $user->kv       = 2;
        $user->save();

        $notify[] = 'KYC data submitted successfully';
        return responseSuccess('kyc_submitted', $notify);
    }

    public function qrCode()
    {
        $notify[] = 'QR Code';
        $user     = auth()->user('agent');
        $qrCode   = $user->qrCode;

        if (!$qrCode) {
            $qrCode              = new QrCode();
            $qrCode->user_id     = $user->id;
            $qrCode->user_type   = 'AGENT';
            $qrCode->unique_code = keyGenerator(15);
            $qrCode->save();
        }
        $uniqueCode = $qrCode->unique_code;
        $qrCode     = cryptoQR($uniqueCode);

        return responseSuccess('qr_code', $notify, [
            'qr_code' => $qrCode,
        ]);
    }

    public function qrCodeDownload()
    {
        $user    = auth()->user('agent');
        $qrCode  = $user->qrCode()->first();
        $general = gs();

        $file     = cryptoQR($qrCode->unique_code);
        $filename = $qrCode->unique_code . '.jpg';

        $manager  = new ImageManager(new Driver());
        $template = $manager->read('assets/images/qr_code_template/' . $general->qr_code_template);

        $client       = new Client();
        $response     = $client->get($file);
        $imageContent = $response->getBody()->getContents();

        $qrCode = $manager->read($imageContent)->cover(2000, 2000);
        $template->place($qrCode, 'center');
        $image = $template->encode();

        $headers = [
            'Content-Type'        => 'image/jpeg',
            'Content-Disposition' => 'attachment; filename=' . $filename,
        ];

        return response()->stream(function () use ($image) {
            echo $image;
        }, 200, $headers);
    }

    public function qrCodeRemove(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file_name' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $file = getFilePath('temporary') . '/' . $request->file_name;

        if (file_exists($file)) {
            unlink($file);
            $notify[] = 'QR code removed successfully';
            return responseSuccess('qr_code_remove', $notify);
        }

        $notify[] = 'Already removed';
        return responseSuccess('qr_code_remove', $notify);
    }

    public function qrCodeScan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $data = QrCode::where('unique_code', $request->code)->first();
        if (!$data) {
            $notify[] = 'QR code doesn\'t match';
            return responseError('validation_error', $notify);
        }

        $notify[] = 'QR code scan';
        return responseSuccess('qr_code_scan', $notify, [
            'user_type'          => $data->user_type,
            'user_data'          => $data->getUser,
            'transaction_charge' => TransactionCharge::get(),
        ]);
    }

    public function commissionLog()
    {
        $notify[] = "Commission Logs";
        $logs     = Transaction::where('user_type', 'AGENT')->where('user_id', auth()->user('agent')->id)->where('remark', 'commission')->apiQuery();

        return responseSuccess('commission_log', $notify, [
            'logs' => $logs,
        ]);
    }

    public function userDataSubmit(Request $request)
    {
        $user = auth()->user('agent');

        if ($user->profile_complete == Status::YES) {
            $notify[] = 'You\'ve already completed your profile';
            return responseError('already_completed', $notify);
        }

        $validator = Validator::make($request->all(), [
            'firstname' => 'required',
            'lastname'  => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user->firstname = $request->firstname;
        $user->lastname  = $request->lastname;
        $user->address   = [
            'country' => @$user->address->country,
            'address' => $request->address,
            'state'   => $request->state,
            'zip'     => $request->zip,
            'city'    => $request->city,
        ];

        $user->profile_complete = Status::YES;
        $user->save();

        $notify[] = 'Profile completed successfully';
        return responseSuccess('profile_completed', $notify);
    }

    public function getDeviceToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $deviceToken = DeviceToken::where('token', $request->token)->where('user_type', 'AGENT')->first();

        if ($deviceToken) {
            $notify[] = 'Already exists';
            return responseSuccess('get_device_token', $notify);
        }

        $deviceToken            = new DeviceToken();
        $deviceToken->user_id   = auth()->user('agent')->id;
        $deviceToken->user_type = 'AGENT';
        $deviceToken->token     = $request->token;
        $deviceToken->is_app    = Status::YES;
        $deviceToken->save();

        $notify[] = 'Token save successfully';
        return responseSuccess('get_device_token', $notify);
    }

    public function show2faData()
    {
        $general   = gs();
        $ga        = new GoogleAuthenticator();
        $user      = auth()->user();
        $secret    = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . $general->site_name, $secret);

        $notify[] = 'Two factor';
        return responseSuccess('two_factor', $notify, [
            'qr_code_url' => $qrCodeUrl,
            'secret'      => $secret,
        ]);
    }

    public function create2fa(Request $request)
    {
        $user      = auth()->user();
        $validator = Validator::make($request->all(), [
            'key'  => 'required',
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts  = 1;
            $user->save();

            $notify[] = 'Google authenticator activated successfully';

            return response()->json([
                'remark'  => '2fa_activated',
                'status'  => 'success',
                'message' => ['success' => $notify],
            ]);
        } else {
            $notify[] = 'Wrong verification code';
            return responseError('wrong_code', $notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user     = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts  = 0;
            $user->save();

            $notify[] = 'Two factor authenticator deactivated successfully';
            return responseSuccess('2fa_deactivated', $notify);
        } else {
            $notify[] = 'Wrong verification code';
            return responseError('wrong_code', $notify);
        }
    }

    public function accountDelete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return responseError('validation_error', $validator->errors());
        }

        $user = auth()->user('agent');

        if (!Hash::check($request->password, $user->password)) {
            $notify[] = 'Invalid Password';
            return responseError('invalid_password', $notify);
        }

        $user->is_deleted = 1;
        $user->save();

        $user->tokens()->delete();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->user_type = 'AGENT';
        $adminNotification->title     = $user->username . ' deleted his account.';
        $adminNotification->click_url = urlPath('admin.agents.detail', $user->id);
        $adminNotification->save();

        $notify[] = 'Account deleted successfully';
        return responseSuccess('account_deleted', $notify);
    }
}
