<?php

namespace App\Http\Controllers\Agent;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\UserActionProcess;
use App\Models\Transaction;
use App\Models\TransactionCharge;
use App\Models\User;
use App\Models\UserAction;
use Illuminate\Http\Request;

class CashInController extends Controller
{

    public function checkUser(Request $request)
    {
        $exist['data'] = User::where(function ($query) use ($request) {
            $query->where('username', $request->user)->orWhere('mobile', $request->user);
        })->where('profile_complete', 1)->exists();
        return response($exist);
    }

    public function cashInForm()
    {
        $pageTitle    = "Cash In";
        $cashInCharge = TransactionCharge::where('slug', 'cash_in')->first();
        return view('Template::agent.cash_in.form', compact('pageTitle', 'cashInCharge'));
    }

    public function confirmCashIn(Request $request)
    {
        $request->validate([
            'amount'   => 'required|gt:0',
            'user'     => 'required',
            'otp_type' => otpType(validation: true),
        ]);

        $cashInCharge = TransactionCharge::where('slug', 'cash_in')->firstOrFail();
        $agent        = agent();
        $userType     = 'AGENT';

        if ($cashInCharge->daily_limit != -1 && $agent->trxLimit('cash_out')['daily'] > $cashInCharge->daily_limit) {
            $notify[] = ['error', 'Your daily cash in limit exceeded'];
            return back()->withNotify($notify)->withInput();
        }

        if ($cashInCharge->monthly_limit != 1 && $agent->trxLimit('cash_out')['monthly'] > $cashInCharge->monthly_limit) {
            $notify[] = ['error', 'Your monthly cash in limit exceeded'];
            return back()->withNotify($notify)->withInput();
        }

        $user = User::where('username', $request->user)->orWhere('email', $request->user)->first();
        if (!$user) {
            $notify[] = ['error', 'Sorry! User not found'];
            return back()->withNotify($notify)->withInput();
        }

        if (($request->amount < $cashInCharge->min_limit) || ($request->amount > $cashInCharge->max_limit)) {
            $notify[] = ['error', 'Please Follow the cash in limit'];
            return back()->withNotify($notify)->withInput();
        }

        //Agent commission
        $fixedCommission   = $cashInCharge->agent_commission_fixed;
        $percentCommission = $request->amount * $cashInCharge->agent_commission_percent / 100;
        $totalCommission   = $fixedCommission + $percentCommission;

        if ($request->amount > $agent->balance) {
            $notify[] = ['error', 'Sorry! Insufficient balance'];
            return back()->withNotify($notify)->withInput();
        }

        $recentSimilarCashin = Transaction::where('user_id', $agent->id)
            ->where('remark', 'cash_in')
            ->where('receiver_id', $user->id)
            ->where('amount', $request->amount)
            ->orderBy('id', 'desc')
            ->where('created_at', '>', now()->subMinutes(3))
            ->first();
        if ($recentSimilarCashin) {
            $notify[] = ['error', 'Request cannot be made.Please wait for 3 minutes at least'];
            return back()->withNotify($notify)->withInput();
        }

        $userAction            = new UserActionProcess();
        $userAction->user_id   = $agent->id;
        $userAction->user_type = $userType;
        $userAction->act       = 'cash_in';

        $userAction->details = [
            'user_id'          => $user->id,
            'amount'           => $request->amount,
            'total_commission' => $totalCommission,
            'done_route'       => route('agent.cash.in.done'),
        ];

        if (count(otpType())) {
            $userAction->type = $request->otp_type;
        }
        $userAction->submit();

        return redirect($userAction->next_route);
    }

    public function cashInDone()
    {
        $userAction = UserAction::forAgent()->where('id', session('action_id'))->first();
        if (!$userAction) {
            $notify[] = ['error', 'Sorry! Unable to process'];
            return to_route('agent.cash.in')->withNotify($notify)->withInput();
        }
        $details = $userAction->details;

        $user = User::where('id', $details->user_id)->first();
        if (!$user) {
            $notify[] = ['error', 'Sorry! User not found'];
            return to_route('agent.cash.in')->withNotify($notify)->withInput();
        }

        $agent = agent();
        if (@$details->amount > $agent->balance) {
            $notify[] = ['error', 'Sorry! Insufficient balance'];
            return to_route('agent.cash.in')->withNotify($notify)->withInput();
        }

        $agent->balance -= $details->amount;
        $agent->save();

        $agentTrx                = new Transaction();
        $agentTrx->user_id       = $agent->id;
        $agentTrx->user_type     = 'AGENT';
        $agentTrx->before_charge = $details->amount;
        $agentTrx->amount        = $details->amount;
        $agentTrx->post_balance  = $agent->balance;
        $agentTrx->charge        = 0;
        $agentTrx->charge_type   = '+';
        $agentTrx->trx_type      = '-';
        $agentTrx->remark        = 'cash_in';
        $agentTrx->details       = 'Cash in to';
        $agentTrx->receiver_id   = $user->id;
        $agentTrx->receiver_type = 'USER';
        $agentTrx->trx           = getTrx();
        $agentTrx->save();

        $user->balance += $details->amount;
        $user->save();

        $userTrx                = new Transaction();
        $userTrx->user_id       = $user->id;
        $userTrx->user_type     = 'USER';
        $userTrx->before_charge = $details->amount;
        $userTrx->amount        = $details->amount;
        $userTrx->post_balance  = $user->balance;
        $userTrx->charge        = 0;
        $userTrx->charge_type   = '+';
        $userTrx->trx_type      = '+';
        $userTrx->remark        = 'cash_in';
        $userTrx->details       = 'Cash in from';
        $userTrx->receiver_id   = $agent->id;
        $userTrx->receiver_type = 'AGENT';
        $userTrx->trx           = $agentTrx->trx;
        $userTrx->save();

        generatePoints($userTrx, $user);

        checkUserReward($user, $userTrx);

        if ($details->total_commission) {
            $agent->balance += $details->total_commission;
            $agent->save();

            $commission                = new Transaction();
            $commission->user_id       = $agent->id;
            $commission->user_type     = 'AGENT';
            $commission->before_charge = $details->total_commission;
            $commission->amount        = $details->total_commission;
            $commission->post_balance  = $agent->balance;
            $commission->charge        = 0;
            $commission->charge_type   = '+';
            $commission->trx_type      = '+';
            $commission->remark        = 'commission';
            $commission->details       = 'Cash in commission';
            $commission->trx           = $agentTrx->trx;
            $commission->save();

            //Agent commission
            notify($agent, 'CASH_IN_COMMISSION_AGENT', [
                'amount'     => showAmount($details->amount, currencyFormat: false),
                'commission' => showAmount($details->total_commission, currencyFormat: false),
                'trx'        => $agentTrx->trx,
                'time'       => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
                'balance'    => showAmount($agent->balance, currencyFormat: false),
            ]);
        }

        //To user
        notify($user, 'CASH_IN', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'agent'   => $agent->username,
            'trx'     => $agentTrx->trx,
            'time'    => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($user->balance, currencyFormat: false),
        ]);

        //To agent
        notify($agent, 'CASH_IN_AGENT', [
            'amount'  => showAmount($details->amount, currencyFormat: false),
            'user'    => $user->fullname,
            'trx'     => $agentTrx->trx,
            'time'    => showDateTime($agentTrx->created_at, 'd/M/Y @h:i a'),
            'balance' => showAmount($agent->balance, currencyFormat: false),
        ]);

        $notify[] = ['success', 'Cash in successful'];
        return to_route('agent.cash.in')->withNotify($notify);
    }
}
