<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UtilityBill;
use Illuminate\Http\Request;

class UtilityBillController extends Controller
{
    public function pending()
    {
        $pageTitle = 'Pending Utility Bills';
        $bills     = $this->utilityBillData('pending');
        return view('admin.utility_bill.all', compact('pageTitle', 'bills'));
    }

    public function approved()
    {
        $pageTitle = 'Approved Utility Bills';
        $bills     = $this->utilityBillData('approved');
        return view('admin.utility_bill.all', compact('pageTitle', 'bills'));
    }

    public function rejected()
    {
        $pageTitle = 'Rejected Utility Bills';
        $bills     = $this->utilityBillData('rejected');
        return view('admin.utility_bill.all', compact('pageTitle', 'bills'));
    }

    public function all()
    {
        $pageTitle = 'All Utility Bills';
        $billData  = $this->utilityBillData(summary: true);
        $bills     = $billData['data'];
        $summary   = $billData['summary'];
        $pending   = $summary['pending'];
        $approved  = $summary['approved'];
        $rejected  = $summary['rejected'];
        $all       = $summary['all'];
        return view('admin.utility_bill.all', compact('pageTitle', 'bills', 'pending', 'approved', 'rejected', 'all'));
    }

    private function utilityBillData($scope = null, $summary = false)
    {
        $bills = $scope ? UtilityBill::$scope() : UtilityBill::query();
        $bills = $bills->searchable(['user:username', 'setupUtilityBill:name', 'trx'])->dateFilter()->with('setupUtilityBill', 'user', 'getTrx');

        if (!$summary) {
            return $bills = $bills->orderBy('id', 'DESC')->paginate(getPaginate());
        } else {
            $pending  = clone $bills;
            $approved = clone $bills;
            $rejected = clone $bills;

            $pendingSummary  = $pending->pending()->sum('amount');
            $approvedSummary = $approved->approved()->sum('amount');
            $rejectedSummary = $rejected->rejected()->sum('amount');
            $allSummary      = $bills->sum('amount');

            return [
                'data'    => $bills->orderBy('id', 'desc')->paginate(getPaginate()),
                'summary' => [
                    'pending'  => $pendingSummary,
                    'approved' => $approvedSummary,
                    'rejected' => $rejectedSummary,
                    'all'      => $allSummary,
                ],
            ];
        }
    }

    public function changeStatus(Request $request)
    {

        $request->validate([
            'id'      => 'required',
            'status'  => 'required|in:1,2',
            'message' => 'required_if:status,==,2',
        ]);

        $utilityBill      = UtilityBill::where('status', Status::PENDING)->findOrFail($request->id);
        $setupUtilityBill = $utilityBill->setupUtilityBill;
        $getTrx           = $utilityBill->getTrx;
        $user             = User::findOrFail($utilityBill->user_id);

        if ($request->status == Status::APPROVED) {
            $utilityBill->status = Status::APPROVED;
            $utilityBill->save();

            notify($user, 'UTILITY_BILL_APPROVE', [
                'amount'       => showAmount($utilityBill->amount, currencyFormat: false),
                'charge'       => showAmount($getTrx->charge, currencyFormat: false),
                'utility'      => $setupUtilityBill->name,
                'trx'          => $getTrx->trx,
                'time'         => showDateTime($utilityBill->created_at, 'd/M/Y @h:i a'),
                'post_balance' => showAmount($getTrx->post_balance, currencyFormat: false),
            ]);

            $notify[] = ['success', 'Utility bill has been approved successfully'];
            return back()->withNotify($notify);
        }

        $utilityBill->status         = Status::REJECTED;
        $utilityBill->admin_feedback = $request->message;
        $utilityBill->save();

        $user->balance += $getTrx->amount;
        $user->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $getTrx->amount;
        $transaction->amount        = $getTrx->amount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = 0;
        $transaction->charge_type   = '-';
        $transaction->trx_type      = '+';
        $transaction->remark        = 'reject_utility_bill';
        $transaction->details       = 'Rejection of utility bill';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = getTrx();
        $transaction->save();

        notify($user, 'UTILITY_BILL_REJECT', [
            'amount'       => showAmount($utilityBill->amount, currencyFormat: false),
            'charge'       => showAmount($getTrx->charge, currencyFormat: false),
            'utility'      => $setupUtilityBill->name,
            'trx'          => $getTrx->trx,
            'reason'       => $request->message,
            'time'         => showDateTime($utilityBill->created_at, 'd/M/Y @h:i a'),
            'post_balance' => showAmount($getTrx->post_balance, currencyFormat: false),
        ]);

        $notify[] = ['success', 'Utility bil has been rejected successfully'];
        return back()->withNotify($notify);
    }
}
