<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DpsPlan;
use App\Models\Organization;
use App\Models\SavingInterval;
use App\Models\SavingTenure;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;

class SavingsOrganizationController extends Controller
{
    public function all()
    {
        $pageTitle = 'Organizations';
        $organizations = Organization::searchable(['name'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.savings.organizations', compact('pageTitle', 'organizations'));
    }

    public function add(Request $request, $id = 0)
    {
        $imgValidation = $id ? 'nullable' : 'required';
        $request->validate([
            'name'  => 'required|string|max:255',
            'image' => [$imgValidation, new FileTypeValidate(['png', 'jpg', 'jpeg'])],
        ]);

        if ($id) {
            $organization = Organization::findOrFail($id);
            $message      = 'Savings organization updated successfully';
        } else {
            $organization = new Organization();
            $message      = 'Savings organization added successfully';
        }

        $organization->name = $request->name;

        if ($request->hasFile('image')) {
            try {
                $old = $organization->image;
                $organization->image = fileUploader($request->image, getFilePath('org'), getFileSize('org'), $old);
            } catch (\Exception $e) {
                $notify[] = ['error', 'Image could not be uploaded'];
                return back()->withNotify($notify);
            }
        }

        $organization->save();

        $notify[] = ['success', $message];
        return back()->withNotify($notify);
    }

    public function status($id)
    {
        return Organization::changeStatus($id);
    }

    public function dpsPlans($id)
    {
        $organization = Organization::active()->findOrFail($id);
        $pageTitle    = 'DPS Plans of ' . $organization->name;
        $tenures      = SavingTenure::active()->orderBy('duration', 'asc')->get();
        $intervals    = SavingInterval::active()->orderBy('installment_interval', 'asc')->get();

        $dpsPlans     = DpsPlan::where('organization_id', $organization->id)
            ->with('savingTenure', 'savingInterval')
            ->orderBy('id', 'desc')
            ->get();

        return view('admin.savings.dps_plan.index', compact('pageTitle', 'dpsPlans', 'organization', 'tenures', 'intervals'));
    }

    public function addDpsPlan($id)
    {
        $organization     = Organization::active()->findOrFail($id);
        $pageTitle        = 'Add DPS Plan for ' . $organization->name;
        $tenures          = SavingTenure::active()->orderBy('duration', 'asc')->get();
        $intervals        = SavingInterval::active()->orderBy('installment_interval', 'asc')->get();

        return view('admin.savings.dps_plan.create', compact('pageTitle', 'organization', 'tenures', 'intervals'));
    }

    public function storeDpsPlan(Request $request, $id)
    {
        $request->validate([
            'tenure_id'            => 'required|integer',
            'interval_id'          => 'required|integer',
            'installment_amount'   => 'required|array',
            'installment_amount.*' => 'required|numeric|gt:0',
            'maturity_amount'      => 'required|array',
            'maturity_amount.*'    => 'required|numeric|gt:0',
            'interest_rate'        => 'required|array',
            'interest_rate.*'      => 'required|numeric|gt:0',
            'penalty_rate'         => 'required|numeric',
            'penalty_interval_id'  => 'required|integer'
        ]);

        $organization = Organization::active()->find($id);
        if (!$organization) {
            $notify[] = ['error', 'Organization not found'];
            return back()->withNotify($notify);
        }

        if (count($request->installment_amount) != count($request->maturity_amount)) {
            $notify[] = ['error', 'Installment and maturity amount mismatched'];
            return back()->withNotify($notify);
        }

        $tenure = SavingTenure::active()->find($request->tenure_id);
        if (!$tenure) {
            $notify[] = ['error', 'Tenure not found'];
            return back()->withNotify($notify);
        }

        $interval = SavingInterval::active()->find($request->interval_id);
        if (!$interval) {
            $notify[] = ['error', 'Interval not found'];
            return back()->withNotify($notify);
        }

        $penaltyInterval = SavingInterval::active()->find($request->penalty_interval_id);
        if (!$penaltyInterval) {
            $notify[] = ['error', 'Penalty interval not found'];
            return back()->withNotify($notify);
        }

        $dpsPlans = [];

        foreach ($request->installment_amount as $key => $installment) {
            if (!isset($request->maturity_amount[$key])) {
                $notify[] = ['error', 'Maturity amount not found'];
                return back()->withNotify($notify);
            }

            $interest       = $request->interest_rate[$key];
            $maturity       = $request->maturity_amount[$key];
            $investedAmount = ($tenure->duration / $interval->installment_interval) * $installment;

            if ($investedAmount >= $maturity) {
                $notify[] = ['error', 'Maturity amount must be greater than invested amount'];
                return back()->withNotify($notify);
            }

            $dpsPlans[] = [
                'organization_id'     => $id,
                'saving_tenure_id'    => $tenure->id,
                'saving_interval_id'  => $interval->id,
                'installment_amount'  => $installment,
                'maturity_amount'     => $maturity,
                'interest_rate'       => $interest,
                'penalty_rate'        => $request->penalty_rate,
                'penalty_interval_id' => $penaltyInterval->id,
                'created_at'          => now(),
                'updated_at'          => now(),
            ];
        }

        DpsPlan::insert($dpsPlans);
        $notify[] = ['success', 'DPS Plan added successfully'];

        return back()->withNotify($notify);
    }

    public function updateDpsPlan(Request $request, $id)
    {
        $request->validate([
            'saving_tenure_id'     => 'required|integer',
            'saving_interval_id'   => 'required|integer',
            'installment_amount'   => 'required|numeric|gt:0',
            'maturity_amount'      => 'required|numeric|gt:0',
            'interest_rate'        => 'required|numeric',
            'penalty_rate'         => 'required|numeric',
            'penalty_interval_id'  => 'required|integer'
        ]);

        $dpsPlan = DpsPlan::findOrFail($id);

        $tenure = SavingTenure::find($request->saving_tenure_id);
        if (!$tenure) {
            $notify[] = ['error', 'Tenure not found'];
            return back()->withNotify($notify);
        }

        $interval = SavingInterval::find($request->saving_interval_id);
        if (!$interval) {
            $notify[] = ['error', 'Interval not found'];
            return back()->withNotify($notify);
        }

        $penaltyInterval = SavingInterval::find($request->penalty_interval_id);
        if (!$penaltyInterval) {
            $notify[] = ['error', 'Penalty interval not found'];
            return back()->withNotify($notify);
        }

        if ($dpsPlan->installment_amount >= $request->maturity_amount) {
            $notify[] = ['error', 'Maturity amount must be greater than invested amount'];
            return back()->withNotify($notify);
        }

        $investmentAmount = ($tenure->duration / $interval->installment_interval) * $request->installment_amount;

        if ($investmentAmount >= $request->maturity_amount) {
            $notify[] = ['error', 'Maturity amount must be greater than invested amount'];
            return back()->withNotify($notify);
        }

        $dpsPlan->saving_tenure_id     = $request->saving_tenure_id;
        $dpsPlan->saving_interval_id   = $request->saving_interval_id;
        $dpsPlan->installment_amount   = $request->installment_amount;
        $dpsPlan->maturity_amount      = $request->maturity_amount;
        $dpsPlan->interest_rate        = $request->interest_rate;
        $dpsPlan->penalty_rate         = $request->penalty_rate;
        $dpsPlan->penalty_interval_id  = $penaltyInterval->id;
        $dpsPlan->save();

        $notify[] = ['success', 'DPS Plan updated successfully'];
        return back()->withNotify($notify);
    }

    public function changeDpsPlanStatus($id)
    {
        return DpsPlan::changeStatus($id);
    }
}
