<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Level;
use App\Models\Reward;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class RewardController extends Controller
{
    public function all()
    {
        $pageTitle   = 'All Rewards';
        $rewards     = Reward::latest('id')->paginate(getPaginate());
        $levels      = Level::active()->orderBy('min_points', 'asc')->get();
        return view('admin.rewards.list', compact('pageTitle', 'rewards', 'levels'));
    }

    public function store(Request $request, $id = 0)
    {
        $request->validate([
            'required_points'        => 'required|numeric|gt:0',
            'name'                   => 'required|string|max:40|unique:rewards,name,' . $id,
            'description'            => 'required|string',
            'reward_value'           => 'required|numeric|gt:0',
            'reward_value_type'      => ['required', Rule::in([Status::FIXED_VALUE, Status::PERCENT_VALUE])],
            'transaction_type'       => 'required|string|in:' . implode(',', allRemarks()),
            'min_transaction_amount' => 'required|numeric|gt:0',
            'validity_days'          => 'required|numeric|gt:0',
        ]);

        $level = Level::active()->findOrFail($request->level_id);

        if ($request->required_points > $level->max_points || $request->required_points < $level->min_points) {
            $notify[] = ['error', 'Required points must be between ' . $level->min_points . ' and ' . $level->max_points];
            return back()->withNotify($notify);
        }

        if ($id) {
            $reward = Reward::findOrFail($id);
            $message = "Reward updated successfully";
        } else {
            $reward = new Reward();
            $message = "Reward added successfully";
        }

        $reward->level_id               = $level->id;
        $reward->required_points        = $request->required_points;
        $reward->name                   = $request->name;
        $reward->description            = $request->description;
        $reward->reward_value           = $request->reward_value;
        $reward->reward_value_type      = $request->reward_value_type;
        $reward->transaction_type       = $request->transaction_type;
        $reward->min_transaction_amount = $request->min_transaction_amount;
        $reward->validity_days          = $request->validity_days;
        $reward->save();

        $notify[] = ['success', $message];
        return back()->withNotify($notify);
    }

    public function status($id)
    {
        return Reward::changeStatus($id);
    }
}
