<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\BankTransfer;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;

class BankTransferController extends Controller
{
    public function pending()
    {
        $pageTitle = 'Pending Bank Transfers';
        $transfers = $this->bankTransferData('pending');
        return view('admin.bank_transfer.all', compact('pageTitle', 'transfers'));
    }

    public function approved()
    {
        $pageTitle = 'Approved Bank Transfers';
        $transfers = $this->bankTransferData('approved');
        return view('admin.bank_transfer.all', compact('pageTitle', 'transfers'));
    }

    public function rejected()
    {
        $pageTitle = 'Rejected Bank Transfers';
        $transfers = $this->bankTransferData('rejected');
        return view('admin.bank_transfer.all', compact('pageTitle', 'transfers'));
    }

    public function all()
    {
        $pageTitle    = 'All Bank Transfers';
        $transferData = $this->bankTransferData(summary: true);
        $transfers    = $transferData['data'];
        $summary      = $transferData['summary'];
        $pending      = $summary['pending'];
        $approved     = $summary['approved'];
        $rejected     = $summary['rejected'];
        $all          = $summary['all'];
        return view('admin.bank_transfer.all', compact('pageTitle', 'transfers', 'pending', 'approved', 'rejected', 'all'));
    }

    private function bankTransferData($scope = null, $summary = false)
    {
        $transfers = $scope ? BankTransfer::$scope() : BankTransfer::query();
        $transfers = $transfers->searchable(['user:username', 'bank:name', 'trx'])->dateFilter()->with('bank', 'user', 'getTrx');

        if (!$summary) {
            return $transfers = $transfers->orderBy('id', 'DESC')->paginate(getPaginate());
        } else {
            $pending  = clone $transfers;
            $approved = clone $transfers;
            $rejected = clone $transfers;

            $pendingSummary  = $pending->pending()->sum('amount');
            $approvedSummary = $approved->approved()->sum('amount');
            $rejectedSummary = $rejected->rejected()->sum('amount');
            $allSummary      = $transfers->sum('amount');

            return [
                'data'    => $transfers->orderBy('id', 'desc')->paginate(getPaginate()),
                'summary' => [
                    'pending'  => $pendingSummary,
                    'approved' => $approvedSummary,
                    'rejected' => $rejectedSummary,
                    'all'      => $allSummary,
                ],
            ];
        }
    }

    public function changeStatus(Request $request)
    {

        $request->validate([
            'id'      => 'required',
            'status'  => 'required|in:1,2',
            'message' => 'required_if:status,==,2',
        ]);

        $bankTransfer = BankTransfer::where('status', Status::PENDING)->findOrFail($request->id);
        $bank         = $bankTransfer->bank;
        $getTrx       = $bankTransfer->getTrx;
        $user         = User::findOrFail($bankTransfer->user_id);

        if ($request->status == Status::APPROVED) {
            $bankTransfer->status = Status::APPROVED;
            $bankTransfer->save();

            notify($user, 'BANK_TRANSFER_APPROVE', [
                'amount'         => showAmount($bankTransfer->amount, currencyFormat: false),
                'charge'         => showAmount($getTrx->charge, currencyFormat: false),
                'account_number' => $bankTransfer->account_number,
                'bank'           => $bank->name,
                'trx'            => $getTrx->trx,
                'time'           => showDateTime($bankTransfer->created_at, 'd/M/Y @h:i a'),
                'post_balance'   => showAmount($getTrx->post_balance, currencyFormat: false),
            ]);

            referralCommission($user, $getTrx->amount, $getTrx->trx, 'bank_transfer');

            $notify[] = ['success', 'Bank transfer has been approved successfully'];
            return back()->withNotify($notify);
        }

        $bankTransfer->status         = Status::REJECTED;
        $bankTransfer->admin_feedback = $request->message;
        $bankTransfer->save();

        $user->balance += $getTrx->amount;
        $user->save();

        $transaction                = new Transaction();
        $transaction->user_id       = $user->id;
        $transaction->user_type     = 'USER';
        $transaction->before_charge = $getTrx->amount;
        $transaction->amount        = $getTrx->amount;
        $transaction->post_balance  = $user->balance;
        $transaction->charge        = 0;
        $transaction->charge_type   = '-';
        $transaction->trx_type      = '+';
        $transaction->remark        = 'reject_bank_transfer';
        $transaction->details       = 'Rejection of bank transfer';
        $transaction->receiver_id   = 0;
        $transaction->receiver_type = null;
        $transaction->trx           = getTrx();
        $transaction->save();

        notify($user, 'BANK_TRANSFER_REJECT', [
            'amount'         => showAmount($bankTransfer->amount, currencyFormat: false),
            'charge'         => showAmount($getTrx->charge, currencyFormat: false),
            'account_number' => $bankTransfer->account_number,
            'bank'           => $bank->name,
            'trx'            => $getTrx->trx,
            'reason'         => $request->message,
            'time'           => showDateTime($bankTransfer->created_at, 'd/M/Y @h:i a'),
            'post_balance'   => showAmount($getTrx->post_balance, currencyFormat: false),
        ]);

        $notify[] = ['success', 'Bank transfer has been rejected successfully'];
        return back()->withNotify($notify);
    }
}
